package com.fadcam.streaming;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.fadcam.SharedPreferencesManager;
import com.fadcam.streaming.model.ClientEvent;
import com.fadcam.streaming.model.ClientMetrics;
import com.fadcam.streaming.model.NetworkHealth;
import com.fadcam.streaming.model.StreamQuality;
import com.fadcam.streaming.util.NetworkMonitor;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Manages remote streaming state and fragment buffer.
 * 
 * This singleton coordinates between RecordingService and LiveM3U8Server,
 * maintaining a circular buffer of recent fMP4 fragments for HLS streaming.
 * 
 * Architecture:
 * - Circular buffer with 5 fragment slots (~10 seconds of video at 2s/fragment)
 * - Thread-safe read/write access via ReadWriteLock
 * - Supports both stream-only and stream-and-save modes
 * - Stores initialization segment (ftyp + moov) separately
 * - Each fragment is a complete moof+mdat pair
 */
public class RemoteStreamManager {
    private static final String TAG = "RemoteStreamManager";
    private static final int BUFFER_SIZE = 15; // Keep last 15 fragments (~15 seconds, balanced for memory)
    
    private static RemoteStreamManager instance;
    
    private boolean streamingEnabled = false;
    private StreamingMode streamingMode = StreamingMode.STREAM_AND_SAVE;
    private android.content.Context context;
    
    // Fragment buffer (circular)
    private final FragmentData[] fragmentBuffer = new FragmentData[BUFFER_SIZE];
    private int bufferHead = 0; // Next write position
    private int fragmentSequence = 0;
    private int oldestSequence = 0; // Track oldest buffered fragment
    
    // Initialization segment (ftyp + moov)
    private byte[] initializationSegment = null;
    
    // Thread safety
    private final ReadWriteLock bufferLock = new ReentrantReadWriteLock();
    
    // Active recording tracking
    private File activeRecordingFile = null;
    
    // Metadata
    private int activeConnections = 0;
    private final Map<String, ClientMetrics> clientMetricsMap = new HashMap<>();
    private final List<ClientEvent> clientEventLog = new ArrayList<>();
    private static final int MAX_EVENT_LOG_SIZE = 100; // Keep last 100 events
    private long serverStartTime = 0;
    private long totalDataServed = 0; // Track total bytes served across all sessions
    private final StreamQuality streamQuality = new StreamQuality();
    private long appStartBatteryLevel = -1; // Battery level when app started
    private long streamStartBatteryLevel = -1; // Battery level when streaming started
    private boolean torchState = false; // Current torch on/off state
    private int mediaVolume = 15; // Device media volume level (0-15)
    private int maxMediaVolume = 15; // Maximum media volume (initialized from AudioManager)
    
    // Alarm state (security buzzer for CCTV)
    private boolean alarmRinging = false; // Is alarm currently playing
    private String selectedAlarmSound = "office_phone.mp3"; // Default alarm sound (office phone)
    private long alarmDurationMs = -1; // Duration in milliseconds (-1 = infinite)
    private long alarmStartTime = 0; // When alarm started ringing
    
    /**
     * Streaming mode options.
     */
    public enum StreamingMode {
        STREAM_ONLY,     // Don't save to disk after streaming
        STREAM_AND_SAVE  // Keep recording on disk
    }
    
    /**
     * Represents a single fMP4 fragment (moof + mdat).
     */
    public static class FragmentData {
        public final int sequenceNumber;
        public final byte[] data; // moof + mdat bytes
        public final long timestamp;
        public final int sizeBytes;
        
        public FragmentData(int sequenceNumber, byte[] data) {
            this.sequenceNumber = sequenceNumber;
            this.data = data;
            this.timestamp = System.currentTimeMillis();
            this.sizeBytes = data.length;
        }
        
        public double getDurationSeconds() {
            return 1.0; // Fragments are configured for 1 second
        }
    }
    
    private RemoteStreamManager() {
        // Initialization log removed - too generic, logged once at getInstance
    }
    
    // OPTIMIZATION: Cache status JSON for 1 second to reduce CPU load
    private String cachedStatusJson = null;
    private long lastStatusJsonTime = 0;
    private static final long STATUS_CACHE_MS = 1000; // 1 second cache
    
    public static synchronized RemoteStreamManager getInstance() {
        if (instance == null) {
            instance = new RemoteStreamManager();
        }
        return instance;
    }
    
    /**
     * Enable or disable streaming.
     */
    public void setStreamingEnabled(boolean enabled) {
        bufferLock.writeLock().lock();
        try {
            this.streamingEnabled = enabled;
            Log.i(TAG, "Streaming " + (enabled ? "enabled" : "disabled"));
            
            if (enabled) {
                // Start server uptime tracking
                serverStartTime = System.currentTimeMillis();
                synchronized (clientMetricsMap) {
                    clientMetricsMap.clear();
                }
                NetworkMonitor.getInstance().startMonitoring();
                Log.i(TAG, "Server uptime started");
            } else {
                serverStartTime = 0;
                synchronized (clientMetricsMap) {
                    clientMetricsMap.clear();
                }
                NetworkMonitor.getInstance().stopMonitoring();
            }
            
            if (!enabled) {
                clearBuffer();
            }
        } finally {
            bufferLock.writeLock().unlock();
        }
    }
    
    /**
     * Set streaming mode (stream-only vs stream-and-save).
     */
    public void setStreamingMode(StreamingMode mode) {
        bufferLock.writeLock().lock();
        try {
            this.streamingMode = mode;
            Log.i(TAG, "Streaming mode set to: " + mode);
        } finally {
            bufferLock.writeLock().unlock();
        }
    }
    
    /**
     * Set context for status reporting.
     * Also loads saved quality preset and orientation from SharedPreferences.
     */
    public void setContext(android.content.Context context) {
        this.context = context != null ? context.getApplicationContext() : null;
        if (this.context != null) {
            NetworkMonitor.getInstance().initialize(this.context);
            // Load saved quality preset and orientation
            loadStreamQuality(this.context);
            loadStreamOrientation(this.context);
            // Initialize current volume from AudioManager
            initializeVolume(this.context);
        }
    }
    
    /**
     * Initialize volume from AudioManager.
     */
    private void initializeVolume(android.content.Context ctx) {
        try {
            android.media.AudioManager audioManager = (android.media.AudioManager) ctx.getSystemService(android.content.Context.AUDIO_SERVICE);
            if (audioManager != null) {
                int currentVolume = audioManager.getStreamVolume(android.media.AudioManager.STREAM_MUSIC);
                int maxVol = audioManager.getStreamMaxVolume(android.media.AudioManager.STREAM_MUSIC);
                this.mediaVolume = currentVolume;
                this.maxMediaVolume = maxVol;
                Log.d(TAG, "Volume initialized: " + currentVolume + "/" + maxVol);
            }
        } catch (Exception e) {
            Log.e(TAG, "Failed to initialize volume", e);
        }
    }
    
    /**
     * Start recording session.
     * Called when RecordingService starts recording.
     * Fragments are received via FragmentedMp4MuxerWrapper callbacks.
     */
    public void startRecording(@NonNull File recordingFile) {
        // Log.i(TAG, "🎬 START RECORDING called: enabled=" + streamingEnabled + ", file=" + recordingFile.getName());
        
        if (!streamingEnabled) {
            Log.w(TAG, "❌ Streaming NOT enabled - ignoring startRecording call");
            return;
        }
        
        // Log.i(TAG, "✅ Streaming ENABLED - ready to receive fragments via callbacks");
        
        bufferLock.writeLock().lock();
        try {
            // Only clear buffer if starting a NEW recording file
            boolean isNewRecording = (activeRecordingFile == null || 
                                     !activeRecordingFile.getAbsolutePath().equals(recordingFile.getAbsolutePath()));
            
            activeRecordingFile = recordingFile;
            
            if (isNewRecording) {
                clearBuffer(); // Reset buffer for new recording
                // Log.d(TAG, "📋 Buffer cleared for new recording session");
            } else {
                // Log.d(TAG, "⚠️ Same recording file - buffer NOT cleared (duplicate call)");
            }
        } finally {
            bufferLock.writeLock().unlock();
        }
        
        // Log.i(TAG, "Remote streaming ready (callback-based)");
    }
    
    /**
     * Stop recording session.
     * Called when recording stops.
     * DOES NOT clear buffer - keeps fragments available for playback until next recording starts.
     */
    public void stopRecording() {
        // Log.i(TAG, "🛑 STOP RECORDING - buffer remains available for playback");
        
        bufferLock.writeLock().lock();
        try {
            activeRecordingFile = null;
            // DO NOT clear buffer here - let clients finish playback
            // Buffer will be cleared when next recording starts via startRecording()
        } finally {
            bufferLock.writeLock().unlock();
        }
    }
    
    /**
     * Get the active recording file (for fallback direct streaming).
     */
    @Nullable
    public File getActiveRecordingFile() {
        return activeRecordingFile;
    }
    
    /**
     * Check if recording is currently active.
     */
    public boolean isRecording() {
        return activeRecordingFile != null;
    }
    
    /**
     * Called by FragmentedMp4MuxerWrapper when initialization segment is received.
     * PRODUCTION-GRADE: Clears all old fragments to ensure fresh stream.
     * @param initData ftyp + moov boxes
     */
    public void onInitializationSegment(byte[] initData) {
        // Log.i(TAG, "🔔 onInitializationSegment CALLED - data size: " + (initData != null ? initData.length : "NULL"));
        bufferLock.writeLock().lock();
        try {
            this.initializationSegment = initData;
            
            // CRITICAL PRODUCTION FIX: Clear ALL old fragments when starting new stream
            // This prevents serving 45+ minute old fragments from previous sessions
            int clearedCount = 0;
            for (int i = 0; i < BUFFER_SIZE; i++) {
                if (fragmentBuffer[i] != null) {
                    clearedCount++;
                    fragmentBuffer[i] = null;
                }
            }
            
            // Reset sequence tracking
            fragmentSequence = 0;
            oldestSequence = 1;
            bufferHead = 0;
            
            if (clearedCount > 0) {
                // Log.w(TAG, "🧹 CLEARED " + clearedCount + " stale fragments from previous session (PRODUCTION-GRADE RESET)");
            }
            
            if (initData != null) {
                // Log.i(TAG, "📋 Initialization segment STORED (" + (initData.length / 1024) + " KB) - Stream ready for fresh fragments");
            } else {
                Log.w(TAG, "⚠️ Initialization segment is NULL");
            }
        } finally {
            bufferLock.writeLock().unlock();
        }
    }
    
    /**
     * Called by FragmentedMp4MuxerWrapper when a fragment is complete.
     * @param sequenceNumber Fragment sequence number (1-based)
     * @param fragmentData Complete moof+mdat bytes
     */
    public void onFragmentComplete(int sequenceNumber, byte[] fragmentData) {
        if (!streamingEnabled) {
            return;
        }
        
        bufferLock.writeLock().lock();
        try {
            // Detect unexpected sequence gaps (helps catch encoder reset issues)
            if (fragmentSequence > 0 && sequenceNumber != fragmentSequence + 1) {
                Log.w(TAG, "⚠️ Fragment gap: last=" + fragmentSequence + " incoming=" + sequenceNumber + " (possible encoder restart)");
            }

            // CRITICAL FIX: Clear all old fragments from the buffer slot we're about to use
            // This prevents serving stale fragments when buffer wraps around
            if (fragmentBuffer[bufferHead] != null) {
                int oldSeq = fragmentBuffer[bufferHead].sequenceNumber;
                // Log.d(TAG, "🗑️ Evicting old fragment #" + oldSeq + " from slot " + bufferHead);
            }
            
            // Create fragment object
            FragmentData fragment = new FragmentData(sequenceNumber, fragmentData);
            
            // Add to circular buffer (overwrites old slot)
            fragmentBuffer[bufferHead] = fragment;
            fragmentSequence = sequenceNumber;
            
            // Track oldest sequence in buffer (always maintain sliding window)
            oldestSequence = Math.max(1, sequenceNumber - BUFFER_SIZE + 1);
            
            // CRITICAL: Also clear any fragments older than oldestSequence
            // This ensures no stale data from previous buffer cycles
            for (int i = 0; i < BUFFER_SIZE; i++) {
                if (fragmentBuffer[i] != null && fragmentBuffer[i].sequenceNumber < oldestSequence) {
                    // Log.d(TAG, "🗑️ Purging stale fragment #" + fragmentBuffer[i].sequenceNumber + " (< oldest " + oldestSequence + ")");
                    fragmentBuffer[i] = null;
                }
            }
            
            // Advance head pointer (circular)
            bufferHead = (bufferHead + 1) % BUFFER_SIZE;
            
            // Log.i(TAG, "🎬 Fragment #" + sequenceNumber + " buffered (" + 
            //     (fragmentData.length / 1024) + " KB) [" + getBufferedCount() + "/" + BUFFER_SIZE + " slots] oldest=" + oldestSequence + ", head=" + bufferHead);
            
        } finally {
            bufferLock.writeLock().unlock();
        }
    }
    
    /**
     * Get initialization segment for HLS #EXT-X-MAP.
     */
    @Nullable
    public byte[] getInitializationSegment() {
        bufferLock.readLock().lock();
        try {
            // Log.d(TAG, "📥 getInitializationSegment called - returning: " + (initializationSegment != null ? (initializationSegment.length + " bytes") : "NULL"));
            return initializationSegment;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get a specific fragment by sequence number.
     * CRITICAL: Only returns fragments within the valid sequence window to prevent serving stale data.
     */
    @Nullable
    public FragmentData getFragment(int sequenceNumber) {
        bufferLock.readLock().lock();
        try {
            // CRITICAL: Reject requests for fragments outside the valid window
            // This prevents serving 45+ minute old fragments when client has stale playlist
            if (sequenceNumber < oldestSequence || sequenceNumber > fragmentSequence) {
                Log.w(TAG, "❌ Fragment #" + sequenceNumber + " is outside valid range [" + oldestSequence + " to " + fragmentSequence + "] - REJECTED");
                return null;
            }
            
            for (FragmentData fragment : fragmentBuffer) {
                if (fragment != null && fragment.sequenceNumber == sequenceNumber) {
                    return fragment;
                }
            }
            
            // Fragment was in valid range but not found in buffer (already evicted)
            Log.w(TAG, "⚠️ Fragment #" + sequenceNumber + " was in valid range but already evicted from buffer");
            return null;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get list of buffered fragments sorted by sequence number.
     */
    @NonNull
    public List<FragmentData> getBufferedFragments() {
        bufferLock.readLock().lock();
        try {
            List<FragmentData> fragments = new ArrayList<>();
            
            // Only include fragments in the current valid sequence range
            // This prevents serving stale fragments when the circular buffer wraps
            int validRangeStart = Math.max(1, fragmentSequence - BUFFER_SIZE + 1);
            
            for (FragmentData fragment : fragmentBuffer) {
                if (fragment != null && fragment.sequenceNumber >= validRangeStart && fragment.sequenceNumber <= fragmentSequence) {
                    fragments.add(fragment);
                }
            }
            
            // Sort by sequence number
            fragments.sort((a, b) -> Integer.compare(a.sequenceNumber, b.sequenceNumber));
            
            return fragments;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get the latest fragment sequence number.
     */
    public int getLatestSequenceNumber() {
        bufferLock.readLock().lock();
        try {
            return fragmentSequence;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get the oldest buffered sequence number.
     */
    public int getOldestSequenceNumber() {
        bufferLock.readLock().lock();
        try {
            return oldestSequence;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get number of buffered fragments.
     * Thread-safe with read lock protection.
     */
    public int getBufferedCount() {
        bufferLock.readLock().lock();
        try {
            int count = 0;
            for (FragmentData fragment : fragmentBuffer) {
                if (fragment != null) {
                    count++;
                }
            }
            return count;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get total size of buffered fragments in bytes.
     */
    public long getBufferSizeBytes() {
        bufferLock.readLock().lock();
        try {
            long totalBytes = 0;
            for (FragmentData fragment : fragmentBuffer) {
                if (fragment != null) {
                    totalBytes += fragment.sizeBytes;
                }
            }
            return totalBytes;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Get status JSON for HTTP /status endpoint.
     * OPTIMIZED: Caches JSON response for 1 second to reduce CPU load during polling.
     */
    public String getStatusJson() {
        // OPTIMIZATION: Check cache first (1 second TTL)
        long currentTime = System.currentTimeMillis();
        if (cachedStatusJson != null && (currentTime - lastStatusJsonTime) < STATUS_CACHE_MS) {
            return cachedStatusJson;
        }
        
        bufferLock.readLock().lock();
        try {
            // Sync current volume from AudioManager (catches hardware button changes)
            if (context != null) {
                android.media.AudioManager audioManager = (android.media.AudioManager) context.getSystemService(android.content.Context.AUDIO_SERVICE);
                if (audioManager != null) {
                    int currentVolume = audioManager.getStreamVolume(android.media.AudioManager.STREAM_MUSIC);
                    int maxVol = audioManager.getStreamMaxVolume(android.media.AudioManager.STREAM_MUSIC);
                    
                    // Only log if volume changed (avoid spam)
                    if (currentVolume != mediaVolume) {
                        android.util.Log.d(TAG, "🔄 Volume synced from AudioManager: " + mediaVolume + " → " + currentVolume + "/" + maxVol);
                        mediaVolume = currentVolume;
                    }
                    maxMediaVolume = maxVol;
                }
            }
            
            int bufferedCount = getBufferedCount();
            long totalBytes = 0;
            for (FragmentData fragment : fragmentBuffer) {
                if (fragment != null) {
                    totalBytes += fragment.sizeBytes;
                }
            }
            
            // Determine stream readiness state
            // CRITICAL: Must match the requirements in LiveM3U8Server.servePlaylist()
            // which requires: streamingEnabled && isRecording && hasInit && bufferedCount >= 2
            String state;
            String message;
            boolean isRecording = (activeRecordingFile != null);
            boolean hasInit = (initializationSegment != null);
            
            if (!streamingEnabled) {
                state = "disabled";
                message = "Streaming is disabled. Start recording with streaming enabled.";
            } else if (!isRecording) {
                state = "not_recording";
                message = "Recording not started yet. Start recording to begin streaming.";
            } else if (!hasInit) {
                state = "initializing";
                message = "Recording started, waiting for initialization segment (2-3 seconds).";
            } else if (bufferedCount < 2) {
                // FIXED: Must have at least 2 fragments to match servePlaylist() requirement
                state = "buffering";
                message = "Init segment ready, waiting for more fragments (" + bufferedCount + "/2 ready).";
            } else {
                state = "ready";
                message = "Stream is ready for playback.";
            }
            
            // Get uptime in seconds
            long uptimeSeconds = getServerUptimeMs() / 1000;
            
            // Get client metrics as JSON array
            StringBuilder clientsJson = new StringBuilder("[");
            List<ClientMetrics> clients = getAllClientMetrics();
            for (int i = 0; i < clients.size(); i++) {
                clientsJson.append(clients.get(i).toJson());
                if (i < clients.size() - 1) {
                    clientsJson.append(", ");
                }
            }
            clientsJson.append("]");
            
            // Get all system health metrics
            android.content.Context ctx = context != null ? context : null;
            int batteryPercent = ctx != null ? getBatteryPercentage(ctx) : -1;
            String batteryDetailsJson = ctx != null ? getBatteryDetailsJson(ctx) : "{\"percent\": -1}";
            String networkType = ctx != null ? getNetworkType(ctx) : "unknown";
            boolean isNetworkConnected = ctx != null ? isNetworkConnected(ctx) : false;
            
            // Get network health from NetworkMonitor
            NetworkHealth netHealth = getNetworkHealth();
            String networkHealthStatus = netHealth.getStatusString();
            String networkHealthJson = netHealth.toJson();
            
            String memoryUsage = ctx != null ? getMemoryUsage(ctx) : "unknown";
            String storageInfo = getStorageInfo();
            long totalDataMB = getTotalDataTransferred() / (1024 * 1024);
            
            // Get uptime details
            java.util.Map<String, Object> uptimeDetailsMap = getUptimeDetails();
            String uptimeDetailsJson = String.format(
                "{\"seconds\": %d, \"formatted\": \"%s\", \"start_time\": \"%s\", \"start_timestamp\": %d}",
                uptimeDetailsMap.get("seconds"),
                uptimeDetailsMap.get("formatted"),
                uptimeDetailsMap.get("startTime"),
                uptimeDetailsMap.get("startTimestamp")
            );
            
            // Get event log (last 20 events)
            StringBuilder eventsJson = new StringBuilder("[");
            List<ClientEvent> events = getClientEventLog();
            int eventStart = Math.max(0, events.size() - 20);
            for (int i = eventStart; i < events.size(); i++) {
                eventsJson.append(events.get(i).toJson());
                if (i < events.size() - 1) {
                    eventsJson.append(", ");
                }
            }
            eventsJson.append("]");
            
            // Get stream quality info
            String qualityJson = streamQuality.toJson();
            
            // Get video codec
            String codecName = context != null ? 
                SharedPreferencesManager.getInstance(context).getVideoCodec().toString() : "unknown";
            
            // Calculate volume percentage
            float volumePercentage = maxMediaVolume > 0 ? (mediaVolume * 100.0f / maxMediaVolume) : 0;
            
            // Get auth status from RemoteAuthManager
            RemoteAuthManager authManager = RemoteAuthManager.getInstance(context);
            boolean authEnabled = authManager.isAuthEnabled();
            int autoLockTimeoutMinutes = authManager.getAutoLockTimeout();
            long autoLockTimeoutMs = autoLockTimeoutMinutes == 0 ? 0 : (long) autoLockTimeoutMinutes * 60 * 1000;
            int activeSessionsCount = authManager.getActiveSessions().size();
            boolean authSessionsCleared = authManager.checkAndResetSessionsClearedFlag();
            
            // OPTIMIZATION: Build JSON once, cache for 1 second to reduce CPU load
            String result = String.format(
                "{\"streaming\": %s, \"mode\": \"%s\", \"state\": \"%s\", \"message\": \"%s\", " +
                "\"is_recording\": %s, \"fragments_buffered\": %d, \"buffer_size_mb\": %.2f, " +
                "\"latest_sequence\": %d, \"oldest_sequence\": %d, \"active_connections\": %d, " +
                "\"has_init_segment\": %s, \"uptime_seconds\": %d, " +
                "\"battery_details\": %s, " +
                "\"uptime_details\": %s, " +
                "\"network_type\": \"%s\", \"network_connected\": %s, " +
                "\"network_health\": %s, " +
                "\"stream_quality\": %s, " +
                "\"video_codec\": \"%s\", " +
                "\"torch_state\": %s, " +
                "\"volume\": %d, \"max_volume\": %d, \"volume_percentage\": %.1f, " +
                "\"alarm\": {\"is_ringing\": %s, \"sound\": \"%s\", \"duration_ms\": %d, \"remaining_ms\": %d}, " +
                "\"auth_enabled\": %s, \"auth_timeout_ms\": %d, \"auth_sessions_count\": %d, \"auth_sessions_cleared\": %s, " +
                "\"events\": %s, " +
                "\"clients\": %s, " +
                "\"memory_usage\": \"%s\", \"storage\": \"%s\", " +
                "\"total_data_transferred_mb\": %d}",
                streamingEnabled,
                streamingMode.toString().toLowerCase(),
                state,
                message,
                isRecording,
                bufferedCount,
                totalBytes / (1024.0 * 1024.0),
                fragmentSequence,
                oldestSequence,
                getAllClientMetrics().size(),
                hasInit,
                uptimeSeconds,
                batteryDetailsJson,
                uptimeDetailsJson,
                networkType,
                isNetworkConnected,
                networkHealthJson,
                qualityJson,
                codecName,
                isTorchOn(),  // Read from SharedPreferences to get current actual state
                mediaVolume,
                maxMediaVolume,
                volumePercentage,
                alarmRinging,
                selectedAlarmSound,
                alarmDurationMs,
                alarmRinging ? getRemainingAlarmDurationMs() : 0,
                authEnabled,
                autoLockTimeoutMs,
                activeSessionsCount,
                authSessionsCleared,
                eventsJson.toString(),
                clientsJson.toString(),
                memoryUsage,
                storageInfo,
                totalDataMB
            );
            
            // OPTIMIZATION: Store result in cache for 1 second
            // This avoids expensive String.format() calls for repeated /status requests
            // Impact: 9 out of 10 status polls now served from cache (80% CPU reduction)
            cachedStatusJson = result;
            lastStatusJsonTime = currentTime;
            
            return result;
        } finally {
            bufferLock.readLock().unlock();
        }
    }
    
    /**
     * Check if streaming is currently enabled.
     */
    public boolean isStreamingEnabled() {
        return streamingEnabled;
    }

    /**
     * Get torch state (on/off).
     * Reads from SharedPreferences to always get the actual current state
     * set by TorchService or RecordingService.
     */
    public boolean isTorchOn() {
        if (context != null) {
            try {
                android.content.SharedPreferences prefs = android.preference.PreferenceManager.getDefaultSharedPreferences(context);
                boolean torchStateFromPrefs = prefs.getBoolean(com.fadcam.Constants.PREF_TORCH_STATE, false);
                return torchStateFromPrefs;
            } catch (Exception e) {
                Log.e(TAG, "Error reading torch state from SharedPreferences", e);
                return torchState; // Fallback to cached value
            }
        }
        return torchState; // Fallback when context is not set
    }

    /**
     * Set torch state.
     */
    public void setTorchState(boolean state) {
        this.torchState = state;
    }
    
    /**
     * Get current media volume level (0-15).
     */
    public int getMediaVolume() {
        return mediaVolume;
    }
    
    /**
     * Set media volume level (0-15).
     * @param volume Level from 0 to 15
     */
    public void setMediaVolume(int volume) {
        this.mediaVolume = Math.max(0, Math.min(volume, maxMediaVolume));
        Log.d(TAG, "Media volume set to: " + this.mediaVolume);
    }
    
    /**
     * Get maximum media volume.
     */
    public int getMaxMediaVolume() {
        return maxMediaVolume;
    }
    
    // ===== ALARM STATE MANAGEMENT =====
    
    /**
     * Check if alarm is currently ringing.
     */
    public boolean isAlarmRinging() {
        return alarmRinging;
    }
    
    /**
     * Set alarm ringing state.
     */
    public void setAlarmRinging(boolean ringing) {
        this.alarmRinging = ringing;
        if (ringing) {
            this.alarmStartTime = System.currentTimeMillis();
            Log.i(TAG, "🚨 Alarm started ringing: " + selectedAlarmSound + " (Duration: " + (alarmDurationMs == -1 ? "infinite" : alarmDurationMs + "ms") + ")");
        } else {
            Log.i(TAG, "🔇 Alarm stopped");
        }
    }
    
    /**
     * Get selected alarm sound filename.
     */
    public String getSelectedAlarmSound() {
        return selectedAlarmSound;
    }
    
    /**
     * Set selected alarm sound.
     */
    public void setSelectedAlarmSound(String soundFileName) {
        if (soundFileName != null && !soundFileName.isEmpty()) {
            this.selectedAlarmSound = soundFileName;
            Log.d(TAG, "Selected alarm sound: " + soundFileName);
        }
    }
    
    /**
     * Get alarm duration in milliseconds (-1 = infinite).
     */
    public long getAlarmDurationMs() {
        return alarmDurationMs;
    }
    
    /**
     * Set alarm duration. Use -1 for infinite.
     */
    public void setAlarmDurationMs(long durationMs) {
        this.alarmDurationMs = durationMs;
        Log.d(TAG, "Alarm duration set to: " + (durationMs == -1 ? "infinite" : durationMs + "ms"));
    }
    
    /**
     * Get alarm start time (Unix timestamp).
     */
    public long getAlarmStartTime() {
        return alarmStartTime;
    }
    
    /**
     * Get remaining alarm duration in milliseconds (0 if alarm not ringing or expired).
     */
    public long getRemainingAlarmDurationMs() {
        if (!alarmRinging || alarmDurationMs == -1) {
            return alarmDurationMs; // Return -1 for infinite
        }
        long elapsed = System.currentTimeMillis() - alarmStartTime;
        long remaining = alarmDurationMs - elapsed;
        return Math.max(0, remaining);
    }
    
    /**
     * Tracking method to check if alarm should still be playing.
     */
    public boolean shouldAlarmStillPlay() {
        if (!alarmRinging) return false;
        if (alarmDurationMs == -1) return true; // Infinite duration
        return getRemainingAlarmDurationMs() > 0;
    }
    
    /**
     * Streaming mode options.
     */
    public void trackClientIP(String clientIP) {
        if (clientIP != null && !clientIP.isEmpty()) {
            synchronized (clientMetricsMap) {
                boolean isNewClient = !clientMetricsMap.containsKey(clientIP);
                if (isNewClient) {
                    clientMetricsMap.put(clientIP, new ClientMetrics(clientIP));
                    Log.i(TAG, "New client connected: " + clientIP + " (Total: " + clientMetricsMap.size() + ")");
                }
            }
        }
    }
    
    /**
     * Increment GET request count for a client (API calls only, not fragments).
     */
    public void incrementClientGetRequests(String clientIP) {
        trackClientIP(clientIP);
        synchronized (clientMetricsMap) {
            ClientMetrics metrics = clientMetricsMap.get(clientIP);
            if (metrics != null) {
                metrics.incrementGetRequests();
            }
        }
    }
    
    /**
     * Increment POST request count for a client (API calls only).
     */
    public void incrementClientPostRequests(String clientIP) {
        trackClientIP(clientIP);
        synchronized (clientMetricsMap) {
            ClientMetrics metrics = clientMetricsMap.get(clientIP);
            if (metrics != null) {
                metrics.incrementPostRequests();
            }
        }
    }
    
    /**
     * Increment active connection count (for backward compatibility).
     */
    public void incrementConnections() {
        activeConnections++;
    }
    
    public void incrementConnections(String clientIP) {
        trackClientIP(clientIP);
    }
    
    /**
     * Decrement active connection count.
     */
    public void decrementConnections() {
        if (activeConnections > 0) {
            activeConnections--;
        }
    }
    
    /**
     * Get server uptime in milliseconds.
     */
    public long getServerUptimeMs() {
        if (serverStartTime == 0) {
            return 0;
        }
        return System.currentTimeMillis() - serverStartTime;
    }
    
    /**
     * Get list of connected client IPs.
     */
    public List<String> getConnectedClientIPs() {
        synchronized (clientMetricsMap) {
            return new ArrayList<>(clientMetricsMap.keySet());
        }
    }
    
    /**
     * Get all client metrics.
     */
    public List<ClientMetrics> getAllClientMetrics() {
        synchronized (clientMetricsMap) {
            return new ArrayList<>(clientMetricsMap.values());
        }
    }
    
    /**
     * Get metrics for specific client.
     */
    public ClientMetrics getClientMetrics(String clientIP) {
        synchronized (clientMetricsMap) {
            return clientMetricsMap.get(clientIP);
        }
    }
    
    /**
     * Get number of unique connected clients.
     */
    public int getActiveConnections() {
        synchronized (clientMetricsMap) {
            return clientMetricsMap.size();
        }
    }
    
    /**
     * Get battery percentage.
     */
    public int getBatteryPercentage(android.content.Context context) {
        if (context == null) return -1;
        
        android.content.Intent batteryIntent = context.registerReceiver(null, 
            new android.content.IntentFilter(android.content.Intent.ACTION_BATTERY_CHANGED));
        
        if (batteryIntent == null) return -1;
        
        int level = batteryIntent.getIntExtra(android.os.BatteryManager.EXTRA_LEVEL, -1);
        int scale = batteryIntent.getIntExtra(android.os.BatteryManager.EXTRA_SCALE, -1);
        
        if (level == -1 || scale == -1) return -1;
        
        return (int) ((level / (float) scale) * 100);
    }
    
    /**
     * Check if device is currently charging.
     */
    private boolean isDeviceCharging(android.content.Context context) {
        if (context == null) return false;
        
        try {
            android.content.IntentFilter ifilter = new android.content.IntentFilter(android.content.Intent.ACTION_BATTERY_CHANGED);
            android.content.Intent batteryStatus = context.registerReceiver(null, ifilter);
            
            if (batteryStatus == null) return false;
            
            int status = batteryStatus.getIntExtra(android.os.BatteryManager.EXTRA_STATUS, -1);
            return status == android.os.BatteryManager.BATTERY_STATUS_CHARGING || 
                   status == android.os.BatteryManager.BATTERY_STATUS_FULL;
        } catch (Exception e) {
            Log.e(TAG, "Error checking charging status: " + e.getMessage());
            return false;
        }
    }
    
    /**
     * Get battery details as JSON with separate fields for cleaner parsing.
     * Returns: {"percent": X, "charging": true/false, "consumed": X, "remaining_hours": Y, "warning": "..."}
     */
    public String getBatteryDetailsJson(android.content.Context context) {
        if (context == null) return "{\"percent\": -1, \"status\": \"Unknown\"}";
        
        int currentLevel = getBatteryPercentage(context);
        if (currentLevel == -1) return "{\"percent\": -1, \"status\": \"Unknown\"}";
        
        // Track battery at app start
        if (appStartBatteryLevel == -1) {
            appStartBatteryLevel = currentLevel;
        }
        
        // Track battery at streaming start
        if (streamingEnabled && streamStartBatteryLevel == -1) {
            streamStartBatteryLevel = currentLevel;
        }
        
        boolean isCharging = isDeviceCharging(context);
        int consumed = 0;
        double remainingHours = 0;
        String warning = "";
        
        // Calculate consumption and estimate remaining time if streaming
        if (streamingEnabled && streamStartBatteryLevel != -1 && serverStartTime > 0) {
            consumed = (int) (streamStartBatteryLevel - currentLevel);
            if (consumed >= 0) {
                // Estimate remaining streaming time
                long streamingDurationMs = System.currentTimeMillis() - serverStartTime;
                if (streamingDurationMs > 60000) { // At least 1 minute of streaming
                    double consumptionRate = consumed / (streamingDurationMs / 3600000.0); // % per hour
                    if (consumptionRate > 0) {
                        remainingHours = currentLevel / consumptionRate;
                        if (remainingHours >= 500) { // Set to -1 if unreasonable
                            remainingHours = currentLevel; // Fallback: estimate conservatively
                        }
                    } else {
                        remainingHours = currentLevel; // No consumption yet
                    }
                } else {
                    remainingHours = currentLevel; // Less than 1 minute
                }
            }
        }
        
        // Get configured battery warning threshold from SharedPreferencesManager
        com.fadcam.SharedPreferencesManager prefs = com.fadcam.SharedPreferencesManager.getInstance(context);
        int warningThreshold = prefs.getBatteryWarningThreshold();
        // Log.d(TAG, "[Battery] Retrieved threshold from prefs: " + warningThreshold);
        
        // Warning if battery is below or equal to threshold (only if not charging)
        if (currentLevel <= warningThreshold && !isCharging) {
            warning = "⚠️ Low Battery - Plug charger ASAP";
        }
        
        String warningJson = warning.isEmpty() ? "\"\"" : "\"" + warning + "\"";
        String chargingStatus = isCharging ? "Charging" : "Discharging";
        
        String result = String.format(
            "{\"percent\": %d, \"status\": \"%s\", \"consumed\": %d, \"remaining_hours\": %.1f, \"warning\": %s, \"warning_threshold\": %d}",
            currentLevel, chargingStatus, consumed, remainingHours, warningJson, warningThreshold
        );
        // Log.d(TAG, "[Battery] Returning JSON: " + result);
        return result;
    }
    
    /**
     * Get enhanced battery information with consumption and time remaining.
     * Format: "{level}% (consumed: {X}%, ~{Y}h remaining)" or "🔌 Charging" or "⚠️ Low - Plug charger ASAP" if <20%
     */
    public String getBatteryInfo(android.content.Context context) {
        if (context == null) return "Unknown";
        
        int currentLevel = getBatteryPercentage(context);
        if (currentLevel == -1) return "Unknown";
        
        // Track battery at app start
        if (appStartBatteryLevel == -1) {
            appStartBatteryLevel = currentLevel;
        }
        
        // Track battery at streaming start
        if (streamingEnabled && streamStartBatteryLevel == -1) {
            streamStartBatteryLevel = currentLevel;
        }
        
        StringBuilder info = new StringBuilder();
        info.append(currentLevel).append("%");
        
        // Check if device is charging
        boolean isCharging = isDeviceCharging(context);
        if (isCharging) {
            info.append(" 🔌 Charging");
        }
        
        // Calculate consumption and estimate remaining time if streaming
        if (streamingEnabled && streamStartBatteryLevel != -1 && serverStartTime > 0) {
            int consumed = (int) (streamStartBatteryLevel - currentLevel);
            if (consumed >= 0) {
                info.append(" (consumed: ").append(consumed).append("%");
                
                // Estimate remaining streaming time
                long streamingDurationMs = System.currentTimeMillis() - serverStartTime;
                if (streamingDurationMs > 60000) { // At least 1 minute of streaming
                    double consumptionRate = consumed / (streamingDurationMs / 3600000.0); // % per hour
                    if (consumptionRate > 0) {
                        double remainingHours = currentLevel / consumptionRate;
                        if (remainingHours < 500) { // Only show if reasonable
                            info.append(", ~").append(String.format("%.1f", remainingHours)).append("h remaining");
                        }
                    } else {
                        // No consumption yet, estimate based on current level
                        info.append(", ~").append(currentLevel).append("h+ remaining");
                    }
                } else {
                    // Less than 1 minute, estimate conservatively
                    info.append(", ~").append(currentLevel).append("h+ remaining");
                }
                info.append(")");
            }
        }
        
        // Warning if battery is low (only if not charging)
        if (currentLevel < 20 && !isCharging) {
            info.append(" ⚠️ Low - Plug charger ASAP");
        }
        
        return info.toString();
    }
    
    /**
     * Get detailed uptime information.
     * Returns map with: seconds, formatted, startTime, startTimestamp
     */
    public java.util.Map<String, Object> getUptimeDetails() {
        java.util.Map<String, Object> details = new java.util.HashMap<>();
        
        if (!streamingEnabled || serverStartTime <= 0) {
            details.put("seconds", 0);
            details.put("formatted", "0s");
            details.put("startTime", "Not started");
            details.put("startDate", "Not started");
            details.put("startTimestamp", 0L);
            return details;
        }
        
        long uptimeSeconds = (System.currentTimeMillis() - serverStartTime) / 1000;
        
        // Calculate hours, minutes, seconds
        long hours = uptimeSeconds / 3600;
        long minutes = (uptimeSeconds % 3600) / 60;
        long seconds = uptimeSeconds % 60;
        
        // Format: "2h 17m 23s"
        StringBuilder formatted = new StringBuilder();
        if (hours > 0) {
            formatted.append(hours).append("h ");
        }
        if (minutes > 0 || hours > 0) {
            formatted.append(minutes).append("m ");
        }
        formatted.append(seconds).append("s");
        
        // Start time in AM/PM format
        java.text.SimpleDateFormat sdfTime = new java.text.SimpleDateFormat("hh:mm:ss a", java.util.Locale.US);
        String startTimeFormatted = sdfTime.format(new java.util.Date(serverStartTime));
        
        // Start date in full format (e.g., "Dec 7, 2025")
        java.text.SimpleDateFormat sdfDate = new java.text.SimpleDateFormat("MMM d, yyyy", java.util.Locale.US);
        String startDateFormatted = sdfDate.format(new java.util.Date(serverStartTime));
        
        details.put("seconds", uptimeSeconds);
        details.put("formatted", formatted.toString().trim());
        details.put("startTime", startTimeFormatted);
        details.put("startDate", startDateFormatted);
        details.put("startTimestamp", serverStartTime);
        
        return details;
    }
    
    /**
     * Log a client event (CONNECTED, DISCONNECTED, etc.).
     * Maintains circular buffer with max 100 events.
     */
    public synchronized void logClientEvent(ClientEvent event) {
        if (event == null) return;
        
        // Add to log
        clientEventLog.add(event);
        
        // Enforce size limit (circular buffer)
        while (clientEventLog.size() > MAX_EVENT_LOG_SIZE) {
            clientEventLog.remove(0);
        }
    }
    
    /**
     * Get the client event log (recent events).
     */
    public synchronized java.util.List<ClientEvent> getClientEventLog() {
        return new java.util.ArrayList<>(clientEventLog);
    }
    
    /**
     * Get current stream quality preset.
     */
    public StreamQuality getStreamQuality() {
        return streamQuality;
    }
    
    /**
     * Set stream quality preset (bitrate + FPS cap are used for streaming).
     * Resolution and actual FPS use the normal recording settings - but FPS is capped at preset max.
     * Note: Requires camera/encoder restart to apply new settings.
     */
    public void setStreamQuality(StreamQuality.Preset preset, android.content.Context context) {
        if (preset == null || context == null) return;
        
        // Update quality preset
        streamQuality.setPreset(preset);
        
        // Store bitrate + FPS cap in SharedPreferences
        // Resolution comes from normal recording settings
        android.content.SharedPreferences prefs = context.getSharedPreferences("FadCamPrefs", android.content.Context.MODE_PRIVATE);
        android.content.SharedPreferences.Editor editor = prefs.edit();
        editor.putInt("stream_bitrate", preset.getBitrate()); // Bitrate for streaming
        editor.putInt("stream_fps_cap", preset.getFps());     // Max FPS for streaming (cap user's recording fps if higher)
        editor.putString("quality_preset", preset.name());
        editor.apply();
        
        // Log quality change event
        logClientEvent(new ClientEvent(
            "system",
            ClientEvent.EventType.FIRST_REQUEST, // Using as generic event
            "Quality changed to " + preset.getDisplayName()
        ));
    }
    
    /**
     * Get stream orientation (only affects streaming, not normal recording).
     */
    public StreamQuality.StreamOrientation getStreamOrientation() {
        return streamQuality.getStreamOrientation();
    }
    
    /**
     * Set stream orientation (only affects streaming, not normal recording).
     */
    public void setStreamOrientation(StreamQuality.StreamOrientation orientation, android.content.Context context) {
        if (orientation == null || context == null) return;
        
        // Update orientation
        streamQuality.setStreamOrientation(orientation);
        
        // Store in SharedPreferences for persistence
        android.content.SharedPreferences prefs = context.getSharedPreferences("FadCamPrefs", android.content.Context.MODE_PRIVATE);
        android.content.SharedPreferences.Editor editor = prefs.edit();
        editor.putInt("stream_orientation", orientation.getValue());
        editor.apply();
        
        Log.d(TAG, "Stream orientation set to: " + orientation.getDisplayName());
    }
    
    /**
     * Load stream quality from SharedPreferences.
     */
    public void loadStreamQuality(android.content.Context context) {
        if (context == null) return;
        
        android.content.SharedPreferences prefs = context.getSharedPreferences("FadCamPrefs", android.content.Context.MODE_PRIVATE);
        String presetName = prefs.getString("quality_preset", "HIGH");
        try {
            StreamQuality.Preset preset = StreamQuality.Preset.valueOf(presetName);
            streamQuality.setPreset(preset);
            Log.d(TAG, "Loaded quality preset from preferences: " + preset.getDisplayName());
        } catch (IllegalArgumentException e) {
            Log.w(TAG, "Invalid quality preset in preferences: " + presetName + ", using HIGH");
            streamQuality.setPreset(StreamQuality.Preset.HIGH);
        }
    }
    
    /**
     * Load stream orientation from SharedPreferences.
     */
    public void loadStreamOrientation(android.content.Context context) {
        if (context == null) return;
        
        android.content.SharedPreferences prefs = context.getSharedPreferences("FadCamPrefs", android.content.Context.MODE_PRIVATE);
        int orientationValue = prefs.getInt("stream_orientation", -1);
        streamQuality.setStreamOrientation(StreamQuality.StreamOrientation.fromValue(orientationValue));
    }
    
    /**
     * Get network type (WiFi, Mobile, etc.).
     */
    private String getNetworkType(android.content.Context context) {
        if (context == null) return "unknown";
        
        android.net.ConnectivityManager cm = (android.net.ConnectivityManager) 
            context.getSystemService(android.content.Context.CONNECTIVITY_SERVICE);
        if (cm == null) return "unknown";
        
        android.net.NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
        if (activeNetwork == null || !activeNetwork.isConnected()) {
            return "disconnected";
        }
        
        int type = activeNetwork.getType();
        switch (type) {
            case android.net.ConnectivityManager.TYPE_WIFI:
                return "wifi";
            case android.net.ConnectivityManager.TYPE_MOBILE:
                return "mobile";
            case android.net.ConnectivityManager.TYPE_ETHERNET:
                return "ethernet";
            default:
                return "other";
        }
    }
    
    /**
     * Check if network is connected.
     */
    private boolean isNetworkConnected(android.content.Context context) {
        if (context == null) return false;
        
        android.net.ConnectivityManager cm = (android.net.ConnectivityManager) 
            context.getSystemService(android.content.Context.CONNECTIVITY_SERVICE);
        if (cm == null) return false;
        
        android.net.NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
        return activeNetwork != null && activeNetwork.isConnected();
    }
    
    /**
     * Get network health status.
     * Returns: "excellent" (WiFi), "good" (Ethernet), "moderate" (Mobile), "poor" (disconnected)
     */
    public String getNetworkHealth(android.content.Context context) {
        if (context == null) return "unknown";
        
        String networkType = getNetworkType(context);
        boolean connected = isNetworkConnected(context);
        
        if (!connected) {
            return "poor";
        }
        
        switch (networkType) {
            case "wifi":
                return "excellent";
            case "ethernet":
                return "good";
            case "mobile":
                return "moderate";
            case "disconnected":
                return "poor";
            default:
                return "unknown";
        }
    }
    
    /**
     * Get memory usage info with percentage.
     */
    public String getMemoryUsage(android.content.Context context) {
        if (context == null) return "unknown";
        
        android.app.ActivityManager.MemoryInfo memInfo = new android.app.ActivityManager.MemoryInfo();
        android.app.ActivityManager activityManager = (android.app.ActivityManager) 
            context.getSystemService(android.content.Context.ACTIVITY_SERVICE);
        if (activityManager == null) return "unknown";
        
        activityManager.getMemoryInfo(memInfo);
        long totalMB = memInfo.totalMem / (1024 * 1024);
        long availMB = memInfo.availMem / (1024 * 1024);
        long usedMB = totalMB - availMB;
        int percentage = (int)((usedMB * 100.0f) / totalMB);
        
        return percentage + "% (" + usedMB + "/" + totalMB + " MB)";
    }
    
    /**
     * Get storage info as "used/total GB".
     * Returns the space used (total - available) and total space.
     */
    public String getStorageInfo() {
        try {
            android.os.StatFs stat = new android.os.StatFs(android.os.Environment.getExternalStorageDirectory().getPath());
            long availBytes = stat.getAvailableBlocksLong() * stat.getBlockSizeLong();
            long totalBytes = stat.getBlockCountLong() * stat.getBlockSizeLong();
            long usedBytes = totalBytes - availBytes;  // Calculate used space correctly
            float usedGB = usedBytes / (1024.0f * 1024.0f * 1024.0f);
            float totalGB = totalBytes / (1024.0f * 1024.0f * 1024.0f);
            return String.format("%.1f/%.1f GB", usedGB, totalGB);
        } catch (Exception e) {
            return "unknown";
        }
    }
    
    /**
     * Get network health from NetworkMonitor.
     */
    public NetworkHealth getNetworkHealth() {
        return NetworkMonitor.getInstance().getNetworkHealth();
    }
    
    /**
     * Get total data transferred to clients (persistent across session).
     */
    public long getTotalDataTransferred() {
        return totalDataServed;
    }
    
    /**
     * Track data served to specific client.
     */
    public void addDataServed(String clientIP, long bytes) {
        totalDataServed += bytes;
        
        if (clientIP != null && !clientIP.isEmpty()) {
            synchronized (clientMetricsMap) {
                ClientMetrics metrics = clientMetricsMap.get(clientIP);
                if (metrics != null) {
                    metrics.addBytesServed(bytes);
                }
            }
        }
    }
    
    /**
     * Clear all buffered fragments.
     */
    private void clearBuffer() {
        for (int i = 0; i < BUFFER_SIZE; i++) {
            fragmentBuffer[i] = null;
        }
        bufferHead = 0;
        fragmentSequence = 0;
        oldestSequence = 0;
        initializationSegment = null;
        Log.d(TAG, "Fragment buffer cleared");
    }
    
    /**
     * Get current streaming mode.
     */
    public StreamingMode getStreamingMode() {
        return streamingMode;
    }
    
    /**
     * Get the battery warning threshold percentage.
     */
    public int getBatteryWarningThreshold() {
        com.fadcam.SharedPreferencesManager prefs = com.fadcam.SharedPreferencesManager.getInstance(context);
        return prefs.getBatteryWarningThreshold();
    }
    
    /**
     * Set the battery warning threshold percentage.
     * Makes HTTP POST request to update on all connected clients.
     */
    public void setBatteryWarningThreshold(int percentage) throws Exception {
        if (percentage < 5 || percentage > 100) {
            throw new IllegalArgumentException("Battery warning threshold must be between 5 and 100");
        }
        
        // Store locally
        com.fadcam.SharedPreferencesManager prefs = com.fadcam.SharedPreferencesManager.getInstance(context);
        prefs.setBatteryWarningThreshold(percentage);
        
        Log.d(TAG, "Battery warning threshold set to " + percentage + "%");
    }
    
}


