package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.templateActivities

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.TemplateActivityType
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.RideLevelSelectorForLists
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TemplateActivityListItem
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ManageTemplateActivitiesDestination
import com.ramcosta.composedestinations.generated.destinations.TemplateActivityAddDestination
import com.ramcosta.composedestinations.generated.destinations.TemplateActivityEditDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@Destination<RootGraph>
@Composable
fun ManageTemplateActivities(
    manageTemplateActivitiesViewModel: ManageTemplateActivitiesViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_manage_template_activities))

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ManageTemplateActivitiesDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Add,
            contentDescription = stringResource(R.string.btn_text_add_template_activity),
            onClick = {
                destinationsNavigator.navigate(TemplateActivityAddDestination)
            }
        ),
        headline = stringResource(R.string.hdr_manage_template_activities),
        advancedFunctionality = advancedFunctionality
    ) {
        var currentRideLevel: Int? by remember { mutableStateOf(null) }

        val templateActivities by manageTemplateActivitiesViewModel.templateActivities.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        val showBuiltInTemplateActivities by manageTemplateActivitiesViewModel.showBuiltInTemplateActivities.collectAsStateWithLifecycle()

        Column(
            modifier = Modifier.fillMaxSize()
        ) {
            TextAndSwitch(
                text = stringResource(R.string.switch_show_built_in_template_activities),
                checked = showBuiltInTemplateActivities
            ) {
                manageTemplateActivitiesViewModel.updateShowBuiltInTemplateActivities(it)
            }
            RideLevelSelectorForLists(
                currentRideLevel,
            ) {
                currentRideLevel = it
            }
            DefaultSpacer()

            // filter template activities by level & built-in or not
            val filteredTemplateActivities: List<TemplateActivity> =
                templateActivities.filter { templateActivity ->
                    currentRideLevel == null || templateActivity.rideLevel == currentRideLevel
                }.filter { templateActivity ->
                    showBuiltInTemplateActivities || templateActivity.typeOfTemplate == TemplateActivityType.CUSTOM
                }
            // lazyColumn those template activities
            LazyColumn(
                modifier = Modifier
                    .fillMaxWidth()
                    .weight(0.7f)
            ) {
                items(filteredTemplateActivities, key = { it.uid }) { templateActivity ->
                    TemplateActivityListItem(templateActivity) {
                        if (templateActivity.typeOfTemplate == TemplateActivityType.CUSTOM) {
                            destinationsNavigator.navigate(
                                TemplateActivityEditDestination(
                                    templateActivity.uid
                                )
                            )
                        } else {
                            // built-in templates can not be edited!
                            // TODO
                            // could be cloned?
                        }
                    }
                }
            }
        }
    }
}
