package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.rides

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Check
import androidx.compose.material3.Icon
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.formatDateAndTimeDifferenceSensibly
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.formatDateAndTimeSensibly
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ViewRideDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@Destination<RootGraph>
@Composable
fun ViewRide(
    rideUid: Long,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_view_ride))

    val viewRideViewModel =
        hiltViewModel<ViewRideViewModel, ViewRideViewModel.ViewRideViewModelFactory> { factory ->
            factory.create(rideUid = rideUid)
        }

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ViewRideDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        headline = stringResource(R.string.hdr_view_ride),
        advancedFunctionality = advancedFunctionality
    ) {
        val ride by viewRideViewModel.ride.collectAsStateWithLifecycle()
        val bike by viewRideViewModel.bike.collectAsStateWithLifecycle()
        val activities by viewRideViewModel.activities.collectAsStateWithLifecycle()

        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
                .padding(8.dp)
        ) {
            if (ride != null) {
                val myRide = ride!!
                Text(
                    text = myRide.name,
                    style = MaterialTheme.typography.headlineMedium
                )
                DefaultSpacer()
                Text(text = stringResource(R.string.lbl_type) + ": " + RideLevel.getLabel(context, myRide.level))
                DefaultSpacer()
                Text(text = stringResource(R.string.lbl_created) + ": ${formatDateAndTimeSensibly(myRide.createdInstant)}")
                DefaultSpacer()
                Text(
                    text = stringResource(R.string.lbl_completed) + ": " + if (myRide.completed) {
                        formatDateAndTimeSensibly(myRide.completedInstant!!)
                    } else {
                        stringResource(R.string.no)
                    }
                )
                if (myRide.completed) {
                    DefaultSpacer()
                    Text(
                        text = stringResource(R.string.lbl_took) + " ${
                            formatDateAndTimeDifferenceSensibly(
                                myRide.createdInstant,
                                myRide.completedInstant!!
                            )
                        }"
                    )
                }
                if (bike != null) {
                    DefaultSpacer()
                    Text(text = stringResource(R.string.lbl_bike_prepared) + ": ${bike?.name}")
                }
                if (activities.isNotEmpty()) {
                    DefaultSpacer()
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.lbl_activities)
                    )
                    DefaultSpacer()
                    activities.forEach { activity ->
                        ListItem(
                            headlineContent = {
                                Text(text = activity.title)
                            },
                            supportingContent = {
                                Text(text = activity.description)
                            },
                            trailingContent = {
                                if (activity.isCompleted) {
                                    Icon(
                                        imageVector = Icons.Default.Check,
                                        contentDescription = stringResource(R.string.done)
                                    )
                                }
                            }
                        )
                    }
                }
            } else {
                // empty ride
                Text(text = stringResource(R.string.text_this_ride_does_not_seem_to_exist))
            }
        }
    }
}