package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.RootData
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.TemplateActivityType
import dagger.hilt.android.lifecycle.HiltViewModel
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.exists
import io.github.vinceglb.filekit.readString
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import javax.inject.Inject
import kotlin.math.max

enum class ImportStateConstants {
    IDLE,
    FILE_SELECTED,
    FILE_ANALYSED,
    IMPORT_FINISHED,
    ERROR
}

data class ImportState(
    val state: ImportStateConstants = ImportStateConstants.IDLE
)

private const val TAG = "ImportDataVM"

@HiltViewModel
class ImportDataViewModel @Inject constructor(
    val repository: KJsRepository
) : ViewModel() {

    private val _importStateFlow = MutableStateFlow(ImportState())
    val importStateFlow: StateFlow<ImportState> = _importStateFlow

    private var _file: MutableStateFlow<PlatformFile?> = MutableStateFlow(null)
    val file: StateFlow<PlatformFile?> = _file

    fun setFile(file: PlatformFile?) {
        if (file != null) {
            _file.value = file
            _importStateFlow.value = ImportState(ImportStateConstants.FILE_SELECTED)
            analyseFile()
        }
    }

    private val _errorMessage: MutableStateFlow<String> = MutableStateFlow("")
    val errorMessage: StateFlow<String> = _errorMessage

    private val _showOverrideControls: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val showOverrideControls: StateFlow<Boolean> = _showOverrideControls

    private val _listOfBikesInFile: MutableStateFlow<List<Bike>> = MutableStateFlow(emptyList())
    val listOfBikesInFile: StateFlow<List<Bike>> = _listOfBikesInFile
    private val _listOfBikesOld: MutableStateFlow<List<Bike>> = MutableStateFlow(emptyList())
    val listOfBikesOld: StateFlow<List<Bike>> = _listOfBikesOld
    private val _listOfBikesNew: MutableStateFlow<List<Bike>> = MutableStateFlow(emptyList())
    val listOfBikesNew: StateFlow<List<Bike>> = _listOfBikesNew
    private val _listOfBikesClashing: MutableStateFlow<List<Bike>> = MutableStateFlow(emptyList())
    val listOfBikesClashing: StateFlow<List<Bike>> = _listOfBikesClashing
    private val _overrideClashingBikes: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val overrideClashingBikes: StateFlow<Boolean> = _overrideClashingBikes

    private val _listOfActivitiesInFile: MutableStateFlow<List<Activity>> =
        MutableStateFlow(emptyList())
    val listOfActivitiesInFile: StateFlow<List<Activity>> = _listOfActivitiesInFile
    private val _listOfActivitiesOld: MutableStateFlow<List<Activity>> =
        MutableStateFlow(emptyList())
    val listOfActivitiesOld: StateFlow<List<Activity>> = _listOfActivitiesOld
    private val _listOfActivitiesNew: MutableStateFlow<List<Activity>> =
        MutableStateFlow(emptyList())
    val listOfActivitiesNew: StateFlow<List<Activity>> = _listOfActivitiesNew
    private val _listOfActivitiesClashing: MutableStateFlow<List<Activity>> =
        MutableStateFlow(emptyList())
    val listOfActivitiesClashing: StateFlow<List<Activity>> = _listOfActivitiesClashing
    private val _overrideClashingActivities: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val overrideClashingActivities: StateFlow<Boolean> = _overrideClashingActivities

    private val _listOfTemplateActivitiesInFile: MutableStateFlow<List<TemplateActivity>> =
        MutableStateFlow(emptyList())
    val listOfTemplateActivitiesInFile: StateFlow<List<TemplateActivity>> =
        _listOfTemplateActivitiesInFile
    private val _listOfTemplateActivitiesOld: MutableStateFlow<List<TemplateActivity>> =
        MutableStateFlow(emptyList())
    val listOfTemplateActivitiesOld: StateFlow<List<TemplateActivity>> =
        _listOfTemplateActivitiesOld
    private val _listOfTemplateActivitiesNew: MutableStateFlow<List<TemplateActivity>> =
        MutableStateFlow(emptyList())
    val listOfTemplateActivitiesNew: StateFlow<List<TemplateActivity>> =
        _listOfTemplateActivitiesNew
    private val _listOfTemplateActivitiesClashing: MutableStateFlow<List<TemplateActivity>> =
        MutableStateFlow(emptyList())
    val listOfTemplateActivitiesClashing: StateFlow<List<TemplateActivity>> =
        _listOfTemplateActivitiesClashing
    private val _overrideClashingTemplateActivities: MutableStateFlow<Boolean> =
        MutableStateFlow(false)
    val overrideClashingTemplateActivities: StateFlow<Boolean> = _overrideClashingTemplateActivities

    private val _listOfComponentsInFile: MutableStateFlow<List<Component>> = MutableStateFlow(
        emptyList()
    )
    val listOfComponentsInFile: StateFlow<List<Component>> = _listOfComponentsInFile
    private val _listOfComponentsOld: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsOld: StateFlow<List<Component>> = _listOfComponentsOld
    private val _listOfComponentsNew: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsNew: StateFlow<List<Component>> = _listOfComponentsNew
    private val _listOfComponentsClashing: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsClashing: StateFlow<List<Component>> = _listOfComponentsClashing
    private val _overrideClashingComponents: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val overrideClashingComponents: StateFlow<Boolean> = _overrideClashingComponents

    private val _listOfRidesInFile: MutableStateFlow<List<Ride>> = MutableStateFlow(emptyList())
    val listOfRidesInFile: StateFlow<List<Ride>> = _listOfRidesInFile
    private val _listOfRidesOld: MutableStateFlow<List<Ride>> = MutableStateFlow(emptyList())
    val listOfRidesOld: StateFlow<List<Ride>> = _listOfRidesOld
    private val _listOfRidesNew: MutableStateFlow<List<Ride>> = MutableStateFlow(emptyList())
    val listOfRidesNew: StateFlow<List<Ride>> = _listOfRidesNew
    private val _listOfRidesClashing: MutableStateFlow<List<Ride>> = MutableStateFlow(emptyList())
    val listOfRidesClashing: StateFlow<List<Ride>> = _listOfRidesClashing
    private val _overrideClashingRides: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val overrideClashingRides: StateFlow<Boolean> = _overrideClashingRides

    private val _includeBuiltInTemplates: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val includeBuiltInTemplates: StateFlow<Boolean> = _includeBuiltInTemplates

    fun setShowOverrideControls(show: Boolean) {
        _showOverrideControls.value = show
    }

    fun setOverrideClashingBikes(override: Boolean) {
        _overrideClashingBikes.value = override
    }

    fun setOverrideClashingActivities(override: Boolean) {
        _overrideClashingActivities.value = override
    }

    fun setOverrideClashingTemplateActivities(override: Boolean) {
        _overrideClashingTemplateActivities.value = override
    }

    fun setOverrideClashingComponents(override: Boolean) {
        _overrideClashingComponents.value = override
    }

    fun setOverrideClashingRides(override: Boolean) {
        _overrideClashingRides.value = override
    }

    fun setIncludeBuiltInTemplates(include: Boolean) {
        _includeBuiltInTemplates.value = include
        analyseFile()
    }

    @OptIn(ExperimentalStdlibApi::class)
    private fun analyseFile() {
        if (file.value != null && file.value!!.exists()) {
            viewModelScope.launch {
                try {
                    var fileContent = file.value!!.readString()
                    Log.d(TAG, "File content: '$fileContent'")
                    // brutal fix for exports from the pre-rides-era
                    if (!fileContent.contains("\"rides\":[")) {
                        // this is a pre-3.2.0 export. Make it work!
                        // TODO this feels very hack-y, I don't like it
                        fileContent = fileContent.dropLast(1)
                        fileContent = fileContent.plus(",\"rides\":[]}")
                        Log.d(TAG, "Fixed: $fileContent")
                        val fileContentEnd =
                            fileContent.substring(max(fileContent.length - 32, 0))
                        Log.d(TAG, "  end: $fileContentEnd")
                    }
                    val newData: RootData? = Json.decodeFromString(fileContent)
                    if (newData != null) {
                        // bikes
                        val newBikes = newData.bikes
                        if (newBikes != null) {
                            // compare with existing
                            _listOfBikesInFile.value = newBikes
                            val oldBikes = repository.getAllBikes()
                            val oldBikeUids: MutableList<Long> = mutableListOf()
                            oldBikes.forEach { oldProcess ->
                                oldBikeUids.add(oldProcess.uid)
                            }
                            _listOfBikesOld.value = emptyList()
                            _listOfBikesClashing.value = emptyList()
                            _listOfBikesNew.value = emptyList()
                            newBikes.forEach { newProcess ->
                                if (oldBikeUids.contains(newProcess.uid)) {
                                    // is it the same?
                                    if (newProcess == repository.getBikeByUid(newProcess.uid)) {
                                        // it is the same. No need to import
                                        val temp = listOfBikesOld.value.toMutableList()
                                        temp.add(newProcess)
                                        _listOfBikesOld.value = temp
                                    } else {
                                        val temp = listOfBikesClashing.value.toMutableList()
                                        temp.add(newProcess)
                                        _listOfBikesClashing.value = temp
                                    }
                                } else {
                                    val temp = listOfBikesNew.value.toMutableList()
                                    temp.add(newProcess)
                                    _listOfBikesNew.value = temp
                                }
                            }
                        }
                        // activities
                        val newActivities = newData.activities
                        if (newActivities != null) {
                            // compare with existing
                            _listOfActivitiesInFile.value = newActivities
                            val oldActivities = repository.getAllActivities()
                            val oldActivityUids: MutableList<Long> = mutableListOf()
                            oldActivities.forEach { oldProcess ->
                                oldActivityUids.add(oldProcess.uid)
                            }
                            _listOfActivitiesOld.value = emptyList()
                            _listOfActivitiesClashing.value = emptyList()
                            _listOfActivitiesNew.value = emptyList()
                            newActivities.forEach { newProcess ->
                                if (oldActivityUids.contains(newProcess.uid)) {
                                    // is it the same?
                                    if (newProcess == repository.getActivityByUid(newProcess.uid)) {
                                        // it is the same. No need to import
                                        val temp = listOfActivitiesOld.value.toMutableList()
                                        temp.add(newProcess)
                                        _listOfActivitiesOld.value = temp
                                    } else {
                                        val temp = listOfActivitiesClashing.value.toMutableList()
                                        temp.add(newProcess)
                                        _listOfActivitiesClashing.value = temp
                                    }
                                } else {
                                    val temp = listOfActivitiesNew.value.toMutableList()
                                    temp.add(newProcess)
                                    _listOfActivitiesNew.value = temp
                                }
                            }
                        }
                        // templates
                        val newTemplates = newData.templateActivities
                        if (newTemplates != null) {
                            // compare with existing
                            _listOfTemplateActivitiesInFile.value = newTemplates
                            val oldTemplateActivities = repository.getAllTemplateActivities()
                            val oldTemplateActivityUids: MutableList<Long> = mutableListOf()
                            oldTemplateActivities.forEach { oldTemplateActivity ->
                                oldTemplateActivityUids.add(oldTemplateActivity.uid)
                            }
                            _listOfTemplateActivitiesOld.value = emptyList()
                            _listOfTemplateActivitiesClashing.value = emptyList()
                            _listOfTemplateActivitiesNew.value = emptyList()
                            newTemplates.filter { template ->
                                template.typeOfTemplate == TemplateActivityType.CUSTOM || includeBuiltInTemplates.value
                            }.forEach { template ->
                                if (oldTemplateActivityUids.contains(template.uid)) {
                                    // is it the same?
                                    if (template == repository.getTemplateActivityByUid(template.uid)) {
                                        // it is the same. No need to import
                                        val temp = listOfTemplateActivitiesOld.value.toMutableList()
                                        temp.add(template)
                                        _listOfTemplateActivitiesOld.value = temp
                                    } else {
                                        val temp =
                                            listOfTemplateActivitiesClashing.value.toMutableList()
                                        temp.add(template)
                                        _listOfTemplateActivitiesClashing.value = temp
                                    }
                                } else {
                                    val temp = listOfTemplateActivitiesNew.value.toMutableList()
                                    temp.add(template)
                                    _listOfTemplateActivitiesNew.value = temp
                                }
                            }
                        }
                        // components
                        val newComponents = newData.components
                        if (newComponents != null) {
                            // compare with existing
                            _listOfComponentsInFile.value = newComponents
                            val oldComponents = repository.getAllComponents()
                            val oldComponentUids: MutableList<Long> = mutableListOf()
                            oldComponents.forEach { oldComponent ->
                                oldComponentUids.add(oldComponent.uid)
                            }
                            _listOfComponentsOld.value = emptyList()
                            _listOfComponentsClashing.value = emptyList()
                            _listOfComponentsNew.value = emptyList()
                            newComponents.forEach { newComponent ->
                                if (oldComponentUids.contains(newComponent.uid)) {
                                    // is it the same?
                                    if (newComponent == repository.getComponentByUid(newComponent.uid)) {
                                        // it is the same. No need to import
                                        val temp = listOfComponentsOld.value.toMutableList()
                                        temp.add(newComponent)
                                        _listOfComponentsOld.value = temp
                                    } else {
                                        val temp = listOfComponentsClashing.value.toMutableList()
                                        temp.add(newComponent)
                                        _listOfComponentsClashing.value = temp
                                    }
                                } else {
                                    val temp = listOfComponentsNew.value.toMutableList()
                                    temp.add(newComponent)
                                    _listOfComponentsNew.value = temp
                                }
                            }
                        }
                        // rides
                        val newRides = newData.rides
                        if (newRides != null) {
                            _listOfRidesInFile.value = newRides
                            val oldRides = repository.getAllRides()
                            val oldRideUids: MutableList<Long> = mutableListOf()
                            oldRides.forEach { oldComponent ->
                                oldRideUids.add(oldComponent.uid)
                            }
                            _listOfRidesOld.value = emptyList()
                            _listOfRidesClashing.value = emptyList()
                            _listOfRidesNew.value = emptyList()
                            newRides.forEach { newRide ->
                                if (oldRideUids.contains(newRide.uid)) {
                                    // exists! Is it the same?
                                    if (newRide == repository.getRideByUid(newRide.uid)) {
                                        val temp = listOfRidesOld.value.toMutableList()
                                        temp.add(newRide)
                                        _listOfRidesOld.value = temp
                                    } else {
                                        val temp = listOfRidesClashing.value.toMutableList()
                                        temp.add(newRide)
                                        _listOfRidesClashing.value = temp
                                    }
                                } else {
                                    val temp = listOfRidesNew.value.toMutableList()
                                    temp.add(newRide)
                                    _listOfRidesNew.value = temp
                                }
                            }
                        }
                    }
                    // done
                    _importStateFlow.value = ImportState(ImportStateConstants.FILE_ANALYSED)
                } catch (exception: Exception) {
                    Log.d(TAG, "Exception: ${exception.message}")
                    _errorMessage.value = exception.message.toString()
                    _importStateFlow.value = ImportState(ImportStateConstants.ERROR)
                }
            }
        }
    }

    fun importNewBikes() {
        if (listOfBikesNew.value.isNotEmpty()) {
            Log.d(TAG, "Importing ${listOfBikesNew.value.size} new bikes...")
            viewModelScope.launch {
                listOfBikesNew.value.forEach { newBike ->
                    repository.insertBike(newBike)
                }
                _listOfBikesNew.value = emptyList()
                Log.d(TAG, "New bikes imported.")
            }
        }
    }

    fun importNewActivities() {
        if (listOfActivitiesNew.value.isNotEmpty()) {
            Log.d(TAG, "Importing ${listOfActivitiesNew.value.size} new activities...")
            viewModelScope.launch {
                listOfActivitiesNew.value.forEach { newActivity ->
                    repository.insertActivity(newActivity)
                }
                _listOfActivitiesNew.value = emptyList()
                Log.d(TAG, "New activities imported.")
            }
        }
    }

    fun importNewTemplateActivities() {
        if (listOfTemplateActivitiesNew.value.isNotEmpty()) {
            Log.d(
                TAG,
                "Importing ${listOfTemplateActivitiesNew.value.size} new template activities..."
            )
            viewModelScope.launch {
                listOfTemplateActivitiesNew.value.forEach { newTemplateActivity ->
                    repository.insertTemplateActivity(newTemplateActivity)
                }
                _listOfTemplateActivitiesNew.value = emptyList()
                Log.d(TAG, "New template activities imported.")
            }
        }
    }

    fun importNewComponents() {
        if (listOfComponentsNew.value.isNotEmpty()) {
            Log.d(TAG, "Importing ${listOfComponentsNew.value.size} new components...")
            viewModelScope.launch {
                listOfComponentsNew.value.forEach { newComponent ->
                    Log.d(TAG, "  Importing component: $newComponent")
                    repository.insertComponent(newComponent)
                }
                _listOfComponentsNew.value = emptyList()
                Log.d(TAG, "New components imported.")
            }
        }
    }

    fun importNewRides() {
        if (listOfRidesNew.value.isNotEmpty()) {
            Log.d(TAG, "Importing ${listOfRidesNew.value.size} new rides...")
            viewModelScope.launch {
                listOfRidesNew.value.forEach { newRide ->
                    Log.d(TAG, "  Importing ride: $newRide")
                    repository.insertRide(newRide)
                }
                _listOfRidesNew.value = emptyList()
                Log.d(TAG, "New rides imported.")
            }
        }
    }

    fun overwriteData() {
        // bikes
        if (overrideClashingBikes.value && listOfBikesClashing.value.isNotEmpty()) {
            Log.d(TAG, "Overwriting ${listOfBikesClashing.value.size} bikes...")
            viewModelScope.launch {
                listOfBikesClashing.value.forEach { clashingBike ->
                    repository.updateBike(clashingBike)
                }
                val tempList = listOfBikesOld.value.toMutableList()
                tempList.addAll(listOfBikesClashing.value)
                _listOfBikesOld.value = tempList
                _listOfBikesClashing.value = emptyList()
                _overrideClashingBikes.value = false
                Log.d(TAG, "Done overwriting bikes.")
            }
        }
        // activities
        if (overrideClashingActivities.value && listOfActivitiesClashing.value.isNotEmpty()) {
            Log.d(TAG, "Overwriting ${listOfActivitiesClashing.value.size} activities...")
            viewModelScope.launch {
                listOfActivitiesClashing.value.forEach { clashingActivity ->
                    repository.updateActivity(clashingActivity)
                }
                val tempList = listOfActivitiesOld.value.toMutableList()
                tempList.addAll(listOfActivitiesClashing.value)
                _listOfActivitiesOld.value = tempList
                _listOfActivitiesClashing.value = emptyList()
                _overrideClashingActivities.value = false
                Log.d(TAG, "Done overwriting activities.")
            }
        }
        // templates
        if (overrideClashingTemplateActivities.value && listOfTemplateActivitiesClashing.value.isNotEmpty()) {
            Log.d(
                TAG,
                "Overwriting ${listOfTemplateActivitiesClashing.value.size} template activities..."
            )
            viewModelScope.launch {
                listOfTemplateActivitiesClashing.value.forEach { clashingTemplateActivity ->
                    repository.updateTemplateActivity(clashingTemplateActivity)
                }
                val tempList = listOfTemplateActivitiesOld.value.toMutableList()
                tempList.addAll(listOfTemplateActivitiesClashing.value)
                _listOfTemplateActivitiesOld.value = tempList
                _listOfTemplateActivitiesClashing.value = emptyList()
                _overrideClashingTemplateActivities.value = false
                Log.d(TAG, "Done overwriting template activities.")
            }
        }
        // components
        if (overrideClashingComponents.value && listOfComponentsClashing.value.isNotEmpty()) {
            Log.d(TAG, "Overwriting ${listOfComponentsClashing.value.size} components...")
            viewModelScope.launch {
                listOfComponentsClashing.value.forEach { clashingComponent ->
                    repository.updateComponent(clashingComponent)
                }
                val tempList = listOfComponentsOld.value.toMutableList()
                tempList.addAll(_listOfComponentsClashing.value)
                _listOfComponentsOld.value = tempList
                _listOfComponentsClashing.value = emptyList()
                _overrideClashingComponents.value = false
                Log.d(TAG, "Done overwriting components")
            }
        }
        // rides
        if (overrideClashingRides.value && listOfRidesClashing.value.isNotEmpty()) {
            Log.d(TAG, "Overwriting ${listOfRidesClashing.value.size} rides...")
            viewModelScope.launch {
                listOfRidesClashing.value.forEach { clashingRide ->
                    repository.updateRide(clashingRide)
                }
                val tempList = listOfRidesOld.value.toMutableList()
                tempList.addAll(listOfRidesClashing.value)
                _listOfRidesOld.value = tempList
                _listOfRidesClashing.value = emptyList()
                _overrideClashingRides.value = false
                Log.d(TAG, "Done overwriting rides")
            }
        }
    }
}
