package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import android.database.sqlite.SQLiteConstraintException
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.ComponentCsv
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.asComponent
import dagger.hilt.android.lifecycle.HiltViewModel
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.exists
import io.github.vinceglb.filekit.source
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.io.IOException
import kotlinx.io.Source
import kotlinx.io.asInputStream
import kotlinx.io.buffered
import org.apache.commons.csv.CSVFormat
import java.io.InputStreamReader
import javax.inject.Inject

private const val TAG = "ICfCSVVM"

@HiltViewModel
class ImportComponentsFromCsvViewModel @Inject constructor(
    val repository: KJsRepository
) : ViewModel() {

    private val _importStateFlow = MutableStateFlow(ImportState())
    val importStateFlow: StateFlow<ImportState> = _importStateFlow

    private val _showOverrideControls: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val showOverrideControls: StateFlow<Boolean> = _showOverrideControls

    fun setShowOverrideControls(show: Boolean) {
        _showOverrideControls.value = show
    }
    fun setOverrideClashingComponents(override: Boolean) {
        _overrideClashingComponents.value = override
    }

    private var _file: MutableStateFlow<PlatformFile?> = MutableStateFlow(null)
    val file: StateFlow<PlatformFile?> = _file

    fun setFile(file: PlatformFile?) {
        if (file != null) {
            _file.value = file
            _importStateFlow.value = ImportState(ImportStateConstants.FILE_SELECTED)
            analyseFile()
        }
    }

    private val _errorMessage: MutableStateFlow<String> = MutableStateFlow("")
    val errorMessage: StateFlow<String> = _errorMessage

    private val _listOfComponentsInFile: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsInFile: StateFlow<List<Component>> = _listOfComponentsInFile
    private val _listOfComponentsOld: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsOld: StateFlow<List<Component>> = _listOfComponentsOld
    private val _listOfComponentsNew: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsNew: StateFlow<List<Component>> = _listOfComponentsNew
    private val _listOfComponentsClashing: MutableStateFlow<List<Component>> =
        MutableStateFlow(emptyList())
    val listOfComponentsClashing: StateFlow<List<Component>> = _listOfComponentsClashing
    private val _overrideClashingComponents: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val overrideClashingComponents: StateFlow<Boolean> = _overrideClashingComponents

    private fun analyseFile() {
        if (file.value != null && file.value!!.exists()) {
            viewModelScope.launch {
                try {
                    val csvFormat = CSVFormat.DEFAULT.builder()
                        .setHeader()
//                        .setSkipHeaderRecord(true)
                        .setIgnoreHeaderCase(true)
                        .setTrim(true)
                        .get()
                    val currentFile = file.value!!
                    val inputSource = currentFile.source()
                    val source: Source = inputSource.buffered()
                    val inputStream = source.asInputStream()
                    val csvParser = csvFormat.parse(InputStreamReader(inputStream))
                    val componentsInFile: MutableList<Component> = mutableListOf()
                    // read line by line and create temporary components
                    csvParser.forEach { csvRecord ->
                        Log.d(TAG, "New record $csvRecord")
                        var csvMap = csvRecord.toMap()
                        ComponentCsv.getHeaders().forEach { headerSet ->
                            csvMap.put(
                                headerSet,
                                csvMap.get(headerSet) ?: csvMap.get(headerSet.lowercase())
                            )
                        }
                        val newComponentCsv = ComponentCsv.createFromMap(csvMap)
                        val newRealComponent = newComponentCsv.asComponent()
                        componentsInFile.add(newRealComponent)
                        // figure out whether we know this one or not
                        val potentialClashingComponent = repository.getComponentByUid(newRealComponent.uid)
                        if (potentialClashingComponent == null) {
                            val tempList = listOfComponentsNew.value.toMutableList()
                            tempList.add(newRealComponent)
                            _listOfComponentsNew.value = tempList
                        } else if (potentialClashingComponent == newRealComponent) {
                            val tempList = listOfComponentsOld.value.toMutableList()
                            tempList.add(newRealComponent)
                            _listOfComponentsOld.value = tempList
                        } else {
                            val tempList = listOfComponentsClashing.value.toMutableList()
                            tempList.add(newRealComponent)
                            _listOfComponentsClashing.value = tempList
                        }
                    }
                    _listOfComponentsInFile.value = componentsInFile
                    Log.d(TAG, "Read ${componentsInFile.size} components.")
                    _importStateFlow.value = ImportState(ImportStateConstants.FILE_ANALYSED)
                } catch (exception: IOException) {
                    Log.d(TAG, "IOException: ${exception.message}")
                    _errorMessage.value = exception.message.toString()
                    _importStateFlow.value = ImportState(ImportStateConstants.ERROR)
                } catch (exception: Exception) {
                    Log.d(TAG, "Exception: ${exception.message}")
                    _errorMessage.value = exception.message.toString()
                    _importStateFlow.value = ImportState(ImportStateConstants.ERROR)
                }
            }
        }
    }

    fun importNewComponents() {
        if (listOfComponentsNew.value.isNotEmpty()) {
            Log.d(TAG, "Importing ${listOfComponentsNew.value.size} new components...")
            viewModelScope.launch {
                val failedImportComponents: MutableList<Component> = mutableListOf()
                listOfComponentsNew.value.forEach { newComponent ->
                    try {
                        Log.d(TAG, "  Importing component: $newComponent")
                        repository.insertComponent(newComponent)
                    } catch (exception: SQLiteConstraintException) {
                        Log.w(TAG, "Oops, component ${newComponent.uid} ${exception.message}")
                        failedImportComponents.add(newComponent)
                    }
                }
                _listOfComponentsNew.value = failedImportComponents
                Log.d(TAG, "New components imported.")
            }
        }
    }

    fun overwriteData() {
        // components
        if (overrideClashingComponents.value && listOfComponentsClashing.value.isNotEmpty()) {
            Log.d(TAG, "Overwriting ${listOfComponentsClashing.value.size} components...")
            viewModelScope.launch {
                listOfComponentsClashing.value.forEach { clashingComponent ->
                    repository.updateComponent(clashingComponent)
                }
                val tempList = listOfComponentsOld.value.toMutableList()
                tempList.addAll(_listOfComponentsClashing.value)
                _listOfComponentsOld.value = tempList
                _listOfComponentsClashing.value = emptyList()
                _overrideClashingComponents.value = false
                Log.d(TAG, "Done overwriting components")
            }
        }
    }
}