package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthOutlinedButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ImportComponentsFromCsvDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import io.github.vinceglb.filekit.dialogs.FileKitMode
import io.github.vinceglb.filekit.dialogs.FileKitType
import io.github.vinceglb.filekit.dialogs.compose.rememberFilePickerLauncher
import io.github.vinceglb.filekit.name

@Destination<RootGraph>
@Composable
fun ImportComponentsFromCsv(
    importComponentsFromCsvViewModel: ImportComponentsFromCsvViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_import_csv))

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ImportComponentsFromCsvDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        headline = stringResource(R.string.hdr_import_csv),
        advancedFunctionality = advancedFunctionality
    ) {
        val importState by importComponentsFromCsvViewModel.importStateFlow.collectAsStateWithLifecycle()

        val errorMessage by importComponentsFromCsvViewModel.errorMessage.collectAsStateWithLifecycle()

        val fileForImport by importComponentsFromCsvViewModel.file.collectAsStateWithLifecycle()

        val launcher = rememberFilePickerLauncher(
            type = FileKitType.File(extensions = listOf("csv")),
            mode = FileKitMode.Single,
            title = stringResource(R.string.pick_a_csv_file)
        ) { file ->
            importComponentsFromCsvViewModel.setFile(file)
        }

        val listOfComponentsInFile by importComponentsFromCsvViewModel.listOfComponentsInFile.collectAsStateWithLifecycle()
        val listOfComponentsOld by importComponentsFromCsvViewModel.listOfComponentsOld.collectAsStateWithLifecycle()
        val listOfComponentsNew by importComponentsFromCsvViewModel.listOfComponentsNew.collectAsStateWithLifecycle()
        val listOfComponentsClashing by importComponentsFromCsvViewModel.listOfComponentsClashing.collectAsStateWithLifecycle()
        val overrideClashingComponents by importComponentsFromCsvViewModel.overrideClashingComponents.collectAsStateWithLifecycle()

        val showOverrides by importComponentsFromCsvViewModel.showOverrideControls.collectAsStateWithLifecycle()

        when (importState.state) {
            // No file selected yet
            ImportStateConstants.IDLE -> {
                Column(
                    modifier = Modifier.padding(8.dp, 0.dp)
                ) {
                    Text(text = stringResource(R.string.start_by_selecting_a_file_to_import_from))
                    DefaultSpacer()
                    FullWidthButton(onClick = {
                        launcher.launch()
                    }) {
                        Text(text = stringResource(R.string.btn_text_select_file))
                    }
                }
            }

            // error
            ImportStateConstants.ERROR -> {
                Column(
                    modifier = Modifier.padding(8.dp, 0.dp)
                ) {
                    Text(text = stringResource(R.string.something_went_wrong))
                    DefaultSpacer()
                    Text(text = errorMessage)
                    DefaultSpacer()
                    Text(text = stringResource(R.string.you_could_try_importing_a_different_file))
                    DefaultSpacer()
                    FullWidthOutlinedButton(onClick = {
                        launcher.launch()
                    }) {
                        Text(text = stringResource(R.string.btn_text_select_different_file))
                    }
                }
            }

            // a file has been selected and/or analysed
            ImportStateConstants.FILE_SELECTED -> {
                Column(
                    modifier = Modifier.padding(8.dp, 0.dp)
                ) {
                    Text(text = stringResource(R.string.file_selected) + fileForImport?.name)
                    FullWidthOutlinedButton(onClick = {
                        launcher.launch()
                    }) {
                        Text(text = stringResource(R.string.btn_text_select_different_file))
                    }
                }
            }

            // a file has been selected and/or analysed
            ImportStateConstants.FILE_ANALYSED -> {
                Column(
                    modifier = Modifier
                        .verticalScroll(rememberScrollState())
                        .padding(8.dp, 0.dp)
                ) {
                    Text(text = stringResource(R.string.file_selected) + fileForImport?.name)
                    FullWidthOutlinedButton(onClick = {
                        launcher.launch()
                    }) {
                        Text(text = stringResource(R.string.btn_text_select_different_file))
                    }

                    // results of analysis
                    Text(
                        modifier = Modifier.padding(horizontal = 0.dp, vertical = 8.dp),
                        text = stringResource(R.string.the_file_contains)
                                + " "
                                + listOfComponentsInFile.size
                                + " "
                                + stringResource(R.string.components)
                    )
                    if (listOfComponentsOld.isNotEmpty() || listOfComponentsClashing.isNotEmpty()) {
                        var text =
                            stringResource(R.string.hdr_components) + ": ${listOfComponentsOld.size} " + stringResource(
                                R.string.already_in_database
                            )
                        if (listOfComponentsClashing.isNotEmpty()) {
                            text += ", ${listOfComponentsClashing.size} " + stringResource(R.string.changed)
                        }
                        text += if (listOfComponentsNew.isEmpty()) {
                            ". " + stringResource(R.string.nothing_to_import)
                        } else {
                            "."
                        }
                        Text(
                            modifier = Modifier.padding(horizontal = 0.dp, vertical = 8.dp),
                            text = text
                        )
                    }
                    if (listOfComponentsNew.isNotEmpty()) {
                        Text(
                            modifier = Modifier
                                .fillMaxWidth()
                                .background(MaterialTheme.colorScheme.primaryContainer)
                                .padding(8.dp),
                            text = stringResource(R.string.hdr_components),
                            style = MaterialTheme.typography.bodyMedium,
                        )
                    }
                    if (listOfComponentsNew.isNotEmpty()) {
                        listOfComponentsNew.forEach {
                            Text(text = it.name)
                        }
                    }
                    if (listOfComponentsNew.isNotEmpty()) {
                        Column(
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            FullWidthButton(onClick = {
                                importComponentsFromCsvViewModel.importNewComponents()
                            }) {
                                Text(stringResource(R.string.btn_text_import_these_components))
                            }
                            DefaultSpacer()
                        }
                    }

                    if (listOfComponentsClashing.isNotEmpty()) {
                        TextAndSwitch(
                            text = stringResource(R.string.overwrite_any),
                            checked = showOverrides
                        ) {
                            importComponentsFromCsvViewModel.setShowOverrideControls(it)
                        }
                    }
                    if (showOverrides && listOfComponentsClashing.isNotEmpty()) {
                        TextAndSwitch(
                            text = stringResource(R.string.overwrite_components),
                            checked = overrideClashingComponents
                        ) {
                            importComponentsFromCsvViewModel.setOverrideClashingComponents(it)
                        }
                    }

                    if (overrideClashingComponents) {
                        FullWidthButton(onClick = {
                            // overwrite!
                            importComponentsFromCsvViewModel.overwriteData()
                        }) {
                            Text(stringResource(R.string.btn_text_overwrite_data))
                        }
                    }
                }
            }

            // finished / done
            ImportStateConstants.IMPORT_FINISHED -> {
                Column(
                    modifier = Modifier.padding(8.dp, 0.dp)
                ) {
                    Text(text = stringResource(R.string.btn_text_import_data))
                    DefaultSpacer()
                    FullWidthButton(onClick = {
                        launcher.launch()
                    }) {
                        Text(text = stringResource(R.string.btn_text_select_different_file))
                    }
                }
            }
        }
    }
}