package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import android.widget.Toast
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.ImportExport
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.TemplateActivityType
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ExportDataDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@OptIn(ExperimentalFoundationApi::class)
@Destination<RootGraph>
@Composable
fun ExportData(
    exportDataViewModel: ExportDataViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_export_data))

    val context = LocalContext.current

    val bikes by exportDataViewModel.allBikes.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val activities by exportDataViewModel.allActivities.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val activitiesButNotForCompletedRides by exportDataViewModel.activitiesButNotForCompletedRides.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val templates by exportDataViewModel.allTemplates.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val components by exportDataViewModel.allComponents.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val rides by exportDataViewModel.allRides.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )

    val includeBuiltInTemplates by exportDataViewModel.exportBuiltInTemplates.collectAsStateWithLifecycle()
    val includeCompletedActivitiesAndRides by exportDataViewModel.exportCompletedActivitiesAndRides.collectAsStateWithLifecycle()

    val numberOfCustomTemplateActivities by exportDataViewModel.numberOfCustomTemplateActivities.collectAsStateWithLifecycle(
        initialValue = 0
    )

    val filteredTemplates = templates.filter { templateActivity ->
        templateActivity.typeOfTemplate == TemplateActivityType.CUSTOM || includeBuiltInTemplates
    }
    val filteredActivities = if (includeCompletedActivitiesAndRides) {
        activities
    } else {
        activitiesButNotForCompletedRides
    }
    val filteredRides = rides.filter { ride ->
        ride.completed == false || includeCompletedActivitiesAndRides
    }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ExportDataDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            enabled = filteredTemplates.size > 0 || bikes.size > 0 || filteredActivities.size > 0 || components.size > 0 || filteredRides.size > 0,
            imageVector = Icons.Default.ImportExport,
            contentDescription = stringResource(R.string.btn_text_export_data),
            onClick = {
                exportDataViewModel.commitExport(
                    context = context,
                    successCallback = {
                        Toast.makeText(
                            context,
                            context.getString(R.string.msg_data_exported_to_downloads_folder),
                            Toast.LENGTH_LONG
                        ).show()
                    }
                )
            }
        ),
        headline = stringResource(R.string.hdr_export_data),
        advancedFunctionality = advancedFunctionality
    ) {
        Column(
            modifier = Modifier
                .fillMaxSize()
        ) {
            Text(
                modifier = Modifier.padding(8.dp, 0.dp),
                text = stringResource(R.string.data_that_will_be_exported)
            )
            TextAndSwitch(
                text = stringResource(R.string.lbl_include_built_in_templates),
                checked = includeBuiltInTemplates
            ) {
                exportDataViewModel.setExportBuiltInTemplates(it)
            }
            TextAndSwitch(
                text = stringResource(R.string.switch_export_completed_activities_rides),
                checked = includeCompletedActivitiesAndRides
            ) {
                exportDataViewModel.setExportCompletedActivitiesAndRides(it)
            }
            LazyColumn {
                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_bikes),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = bikes, key = { "bike-${it.uid}" }) {
                    Text(text = it.name)
                }

                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_activities),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = filteredActivities, key = { "activity-${it.uid}" }) {
                    Text(text = it.title)
                }

                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_rides),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = filteredRides, key = { "ride-${it.uid}" }) {
                    Text(text = it.name)
                }

                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_template_activities),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = filteredTemplates, key = { "template-${it.uid}" }) {
                    Text(
                        text = "${
                            RideLevel.getLabel(
                                context,
                                it.rideLevel
                            )
                        } - ${it.title}"
                    )
                }

                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_components),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = components, key = { "component-${it.uid}" }) {
                    Text(text = it.name)
                }
            }
        }
    }
}
