package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import android.content.ContentValues
import android.content.Context
import android.os.Environment
import android.provider.MediaStore
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.ComponentCsv
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.asStringList
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import org.apache.commons.csv.CSVFormat
import org.apache.commons.csv.CSVPrinter
import java.io.OutputStreamWriter
import javax.inject.Inject
import kotlin.time.Clock

private const val TAG = "ExportCtCVM"

@HiltViewModel
class ExportComponentsToCsvViewModel @Inject constructor(
    private val userPreferencesManager: UserPreferencesManager,
    repository: KJsRepository
) : ViewModel() {

    val allComponents = repository.observeComponents

    val exportRetiredComponents: StateFlow<Boolean> =
        userPreferencesManager.exportRetiredComponents().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )

    fun setExportRetiredComponents(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setExportRetiredComponents(enabled)
        }
    }

    fun commitExport(
        context: Context,
        successCallback: (() -> Unit)? = null,
        failureCallback: ((String) -> Unit)? = null
    ) {
        viewModelScope.launch {
            try {
                // collect the data
                val allData: MutableList<ComponentCsv> = mutableListOf()
                val componentList = allComponents.first()
                val nowInstant = Clock.System.now()
                componentList.forEach { component ->
                    if (exportRetiredComponents.value || component.retirementDate == null || component.retirementDate < nowInstant.toLocalDateTime(
                            TimeZone.currentSystemDefault()
                        ).date
                    ) {
                        val componentCsv = ComponentCsv.createFromComponent(component)
                        allData.add(componentCsv)
                    }
                }
                // prepare for writing it
                val now =
                    Clock.System.now().toLocalDateTime(TimeZone.Companion.currentSystemDefault())
                val timeComponentForExportFileName = now.toString()
                // now write it to the Downloads folder
                val retiredComponent = if (exportRetiredComponents.value) {
                    "all"
                } else {
                    "nonRetired"
                }
                val contentValues = ContentValues().apply {
                    put(
                        MediaStore.MediaColumns.DISPLAY_NAME,
                        "jkbike-components-$retiredComponent-$timeComponentForExportFileName"
                    )
                    put(MediaStore.MediaColumns.MIME_TYPE, "text/csv")
                    put(MediaStore.MediaColumns.RELATIVE_PATH, Environment.DIRECTORY_DOWNLOADS)
                }
                val resolver = context.contentResolver
                val uri = resolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)
                if (uri != null) {
                    resolver.openOutputStream(uri).use { stream ->
                        val writer = OutputStreamWriter(stream)
                        val headerArray = ComponentCsv.getHeaders().toTypedArray()
                        val csvFormat = CSVFormat.DEFAULT.builder()
                            .setHeader(*headerArray)
                            .get()
                        val csvPrinter = CSVPrinter(writer, csvFormat)
                        allData.forEach { csvRecord ->
                            csvPrinter.printRecord(csvRecord.asStringList())
                        }
                        csvPrinter.flush()
                        csvPrinter.close()
                    }
                }
                Log.d(TAG, "Exported successfully")
                // notify
                if (successCallback != null) {
                    successCallback()
                }

            } catch (exception: Exception) {
                Log.d(TAG, "Something untoward happened: ${exception.message}")
                if (failureCallback != null) {
                    failureCallback(exception.message ?: "Failure (unknown)")
                }
            }
        }
    }
}