package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.importExport

import android.widget.Toast
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.ImportExport
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ExportComponentsToCsvDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.Clock

@Destination<RootGraph>
@Composable
fun ExportComponentsToCsv(
    exportComponentsToCsvViewModel: ExportComponentsToCsvViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_export_data))

    val context = LocalContext.current

    val components by exportComponentsToCsvViewModel.allComponents.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )

    val exportRetiredComponents by exportComponentsToCsvViewModel.exportRetiredComponents.collectAsStateWithLifecycle()

    val now = Clock.System.now()
    val filteredComponents = components.filter { component ->
        exportRetiredComponents || component.retirementDate == null || component.retirementDate > now.toLocalDateTime(
            TimeZone.currentSystemDefault()
        ).date
    }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ExportComponentsToCsvDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            enabled = filteredComponents.size > 0,
            imageVector = Icons.Default.ImportExport,
            contentDescription = stringResource(R.string.btn_text_export_data),
            onClick = {
                exportComponentsToCsvViewModel.commitExport(
                    context = context,
                    successCallback = {
                        Toast.makeText(
                            context,
                            context.getString(R.string.msg_data_exported_to_downloads_folder),
                            Toast.LENGTH_LONG
                        ).show()
                    }
                )
            }
        ),
        headline = stringResource(R.string.hdr_export_data),
        advancedFunctionality = advancedFunctionality
    ) {
        Column(
            modifier = Modifier
                .fillMaxSize()
        ) {
            Text(
                modifier = Modifier.padding(8.dp, 0.dp),
                text = stringResource(R.string.data_that_will_be_exported)
            )
            TextAndSwitch(
                text = stringResource(R.string.switch_include_retired_components),
                checked = exportRetiredComponents
            ) {
                exportComponentsToCsvViewModel.setExportRetiredComponents(it)
            }
            LazyColumn {
                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_components),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }
                items(items = filteredComponents, key = { "component-${it.uid}" }) { component ->
                    ListItem(
                        headlineContent = {
                            Text(text = component.name)
                        },
                        supportingContent = {
                            Text(text = component.description)
                        }
                    )
                }
            }
        }
    }
}