package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Dataset
import androidx.compose.material.icons.filled.Download
import androidx.compose.material.icons.filled.Notifications
import androidx.compose.material.icons.filled.NotificationsOff
import androidx.compose.material.icons.filled.QrCode
import androidx.compose.material.icons.filled.Upload
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.core.graphics.toColorInt
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ShelvedComponents
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.IconSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.SingleChoiceSegmentedButtonWithText
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ComponentAddDestination
import com.ramcosta.composedestinations.generated.destinations.ComponentAnalysisDestination
import com.ramcosta.composedestinations.generated.destinations.ComponentEditDestination
import com.ramcosta.composedestinations.generated.destinations.ComponentRemoteEditDestination
import com.ramcosta.composedestinations.generated.destinations.ExportComponentsToCsvDestination
import com.ramcosta.composedestinations.generated.destinations.ImportComponentsFromCsvDestination
import com.ramcosta.composedestinations.generated.destinations.ManageComponentsDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@OptIn(ExperimentalFoundationApi::class)
@Destination<RootGraph>
@Composable
fun ManageComponents(
    bikeUid: Long? = null,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_manage_components))

    val manageComponentsViewModel =
        hiltViewModel<ManageComponentsViewModel, ManageComponentsViewModel.ManageComponentsViewModelFactory> { factory ->
            factory.create(bikeUid = bikeUid)
        }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ManageComponentsDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            ),
            KJsAction(
                imageVector = Icons.Default.Upload,
                contentDescription = stringResource(R.string.import_from_csv),
                onClick = {
                    destinationsNavigator.navigate(ImportComponentsFromCsvDestination)
                }
            ),
            KJsAction(
                imageVector = Icons.Default.Download,
                contentDescription = stringResource(R.string.export_to_csv),
                onClick = {
                    destinationsNavigator.navigate(ExportComponentsToCsvDestination)
                }
            ),
            KJsAction(
                imageVector = Icons.Default.QrCode,
                contentDescription = stringResource(R.string.scan_qr_code),
                onClick = {
                    destinationsNavigator.navigate(ComponentRemoteEditDestination)
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Add,
            contentDescription = stringResource(R.string.btn_text_add_component),
            onClick = {
                destinationsNavigator.navigate(ComponentAddDestination)
            }
        ),
        headline = stringResource(R.string.hdr_manage_components),
        advancedFunctionality = advancedFunctionality
    ) {
        val components: List<Component> by manageComponentsViewModel.components.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val retiredComponents by manageComponentsViewModel.retiredComponents.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val componentsOnShelf: List<ShelvedComponents> by manageComponentsViewModel.componentsOnShelf.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val bikes: List<Bike> by manageComponentsViewModel.bikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val currentBike: Bike? by manageComponentsViewModel.selectedBike.collectAsStateWithLifecycle()
        var componentType by rememberSaveable { mutableStateOf(ComponentType.ON_BIKE) }

        Column(
            modifier = Modifier
                .fillMaxSize()
        ) {
            SingleChoiceSegmentedButtonWithText(
                componentType = componentType,
                initCurrentBike = currentBike,
                bikes = bikes,
                numberOfComponentsOnShelf = componentsOnShelf.size,
                numberOfRetiredComponents = retiredComponents.size,
                onComponentTypeSwitched = { cType ->
                    componentType = cType
                },
                onSelected = { bike ->
                    manageComponentsViewModel.updateSelectedBike(bike?.uid)
                    componentType = ComponentType.ON_BIKE
                },
                onNavigateToAnalysis = {
                    destinationsNavigator.navigate(ComponentAnalysisDestination)
                }
            )
            DefaultSpacer()

            val adaptedComponentsList: List<Component> =
                when (componentType) {
                    ComponentType.ON_SHELF -> {
                        componentsOnShelf.map { componentOnShelf ->
                            Component(
                                name = componentOnShelf.name,
                                description = componentOnShelf.description,
                                acquisitionDate = componentOnShelf.acquisitionDate,
                                firstUseDate = componentOnShelf.firstUseDate,
                                lastCheckDate = componentOnShelf.lastCheckDate,
                                lastCheckMileage = componentOnShelf.lastCheckMileage,
                                currentMileage = componentOnShelf.currentMileage,
                                mileageOffset = componentOnShelf.mileageOffset,
                                bikeUid = componentOnShelf.bikeUid,
                                titleForAutomaticActivities = componentOnShelf.titleForAutomaticActivities,
                                wearLevel = componentOnShelf.wearLevel,
                                retirementDate = componentOnShelf.retirementDate,
                                retirementReason = componentOnShelf.retirementReason,
                                checkIntervalMiles = componentOnShelf.checkIntervalMiles,
                                checkIntervalDays = componentOnShelf.checkIntervalDays,
                                notifyWhenActivityCreated = componentOnShelf.notifyWhenActivityCreated,
                                lastCheckNotificationMileage = componentOnShelf.lastCheckNotificationMileage,
                                lastCheckNotificationDate = componentOnShelf.lastCheckNotificationDate,
                                uid = componentOnShelf.uid,
                            )
                        }
                    }

                    ComponentType.RETIRED -> {
                        retiredComponents.map { retiredComponent ->
                            Component(
                                name = retiredComponent.name,
                                description = retiredComponent.description,
                                acquisitionDate = retiredComponent.acquisitionDate,
                                firstUseDate = retiredComponent.firstUseDate,
                                lastCheckDate = retiredComponent.lastCheckDate,
                                lastCheckMileage = retiredComponent.lastCheckMileage,
                                currentMileage = retiredComponent.currentMileage,
                                mileageOffset = retiredComponent.mileageOffset,
                                bikeUid = retiredComponent.bikeUid,
                                titleForAutomaticActivities = retiredComponent.titleForAutomaticActivities,
                                wearLevel = retiredComponent.wearLevel,
                                retirementDate = retiredComponent.retirementDate,
                                retirementReason = retiredComponent.retirementReason,
                                checkIntervalMiles = retiredComponent.checkIntervalMiles,
                                checkIntervalDays = retiredComponent.checkIntervalDays,
                                notifyWhenActivityCreated = retiredComponent.notifyWhenActivityCreated,
                                lastCheckNotificationMileage = retiredComponent.lastCheckNotificationMileage,
                                lastCheckNotificationDate = retiredComponent.lastCheckNotificationDate,
                                uid = retiredComponent.uid,
                            )
                        }
                    }

                    else -> {
                        components.filter { component ->
                            (currentBike == null && component.bikeUid != null) || (currentBike != null && currentBike!!.uid == component.bikeUid)
                        }
                    }
                }

            LazyColumn(
                modifier = Modifier
                    .fillMaxWidth()
                    .weight(0.7f)
            ) {
                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_components),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }

                items(adaptedComponentsList, key = { "component.${it.uid}" }) { component ->
                    Surface(
                        modifier = Modifier
                            .combinedClickable(
                                onClick = {
                                    destinationsNavigator.navigate(
                                        ComponentEditDestination(
                                            component.uid
                                        )
                                    )
                                },
                            )
                            .padding(4.dp),
                    ) {
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            AnimatedVisibility(
                                visible = component.notifyWhenActivityCreated && componentType == ComponentType.ON_BIKE
                            ) {
                                Icon(
                                    imageVector = Icons.Default.Notifications,
                                    contentDescription = "Notifications on",
                                    tint = Color("#006600".toColorInt()),
                                )
                            }
                            AnimatedVisibility(
                                visible = !component.notifyWhenActivityCreated && componentType == ComponentType.ON_BIKE
                            ) {
                                Icon(
                                    imageVector = Icons.Default.NotificationsOff,
                                    contentDescription = "Notifications off",
                                    tint = Color("#660000".toColorInt()),
                                )
                            }
                            AnimatedVisibility(
                                visible = componentType != ComponentType.ON_BIKE
                            ) {
                                Icon(
                                    imageVector = Icons.Default.Dataset,
                                    contentDescription = stringResource(R.string.component),
                                )
                            }
                            IconSpacer()
                            Text(
                                text = component.name,
                                maxLines = 1,
                                overflow = TextOverflow.Ellipsis
                            )
                            Spacer(modifier = Modifier.weight(0.1f))
                            Text(
                                text = component.description,
                                overflow = TextOverflow.Ellipsis,
                                maxLines = 1,
                            )
                            // TODO may want to show if this wants notifications
                        }
                    }
                }
            }

            DefaultSpacer()
            Text(
                modifier = Modifier.padding(8.dp, 0.dp),
                text = stringResource(R.string.text_you_can_also_manage_components_in_your_browser_on_jkbike_net_mdp_components),
                textAlign = TextAlign.Justify,
                style = MaterialTheme.typography.bodySmall
            )
        }
    }
}