package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import android.content.Context
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.ComponentCsv
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.decodeFromStream
import okhttp3.OkHttpClient
import okhttp3.Request
import java.net.HttpURLConnection
import java.net.URL
import javax.inject.Inject


private const val TAG = "CREVM"

@HiltViewModel
class ComponentRemoteEditViewModel @Inject constructor(
    val userPreferencesManager: UserPreferencesManager,
    val repository: KJsRepository
) : ViewModel() {

    val allComponents = repository.observeComponents

    private val _url: MutableStateFlow<URL> =
        MutableStateFlow(URL("https://jkbike.net/mdp/components/connectAndUpload.php"))
    val url: StateFlow<URL> = _url

    private val _codeIsValid: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val codeIsValid: StateFlow<Boolean> = _codeIsValid

    private val _uploadUrl: MutableStateFlow<URL> =
        MutableStateFlow(URL("https://jkbike.net/mdp/components/connectAndDownload.php"))
    val uploadUrl: StateFlow<URL> = _uploadUrl

    @OptIn(ExperimentalSerializationApi::class)
    fun setCode(
        code: String,
        context: Context
    ) {
        val uri = URL(code)
        _codeIsValid.value = true
        _url.value = uri

        // get components
        viewModelScope.launch {
            try {
                // collect the data
                val allData: MutableList<ComponentCsv> = mutableListOf()
                val componentList = allComponents.first()
                val jsonData = Json.encodeToString(componentList)
                Log.d(TAG, "Json data $jsonData")
                withContext(Dispatchers.IO) {
                    with(uri.openConnection() as HttpURLConnection) {
                        // Enable input and output
                        doOutput = true
                        requestMethod = "POST"
                        setRequestProperty("Content-Type", "application/json; utf-8")

                        // Write the request body
                        outputStream.use { os ->
                            val input = jsonData.toByteArray(charset("utf-8"))
                            os.write(input, 0, input.size)
                        }

                        // Print response code
                        Log.d(TAG, "Response Code: $responseCode")

                        inputStream.use { inputStream ->
                            val potentialUrlString = Json.decodeFromStream<String>(inputStream)
                            Log.d(TAG, "DownloadURL: $potentialUrlString")
                            if (!potentialUrlString.isNullOrEmpty()) {
                                try {
                                    _uploadUrl.value = URL(potentialUrlString)
                                } catch (exception: Exception) {
                                    Log.w(TAG, "Exception: $exception")
                                }
                            }
                        }
                    }
                }
                _uploadSuccessful.value = true
            } catch (exception: Exception) {
                Log.d(TAG, "Exception: $exception")
            }
        }
    }

    private val _uploadSuccessful: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val uploadSuccessful: StateFlow<Boolean> = _uploadSuccessful

    fun pullComponentsBack() {
        val client = OkHttpClient()
        try {
            viewModelScope.launch {
                withContext(Dispatchers.IO) {
                    val request: Request? = Request.Builder()
                        .url(uploadUrl.value)
                        .build()
                    Log.d(TAG, "Request URL: ${uploadUrl.value}")
                    if (request != null) {
                        val response = client.newCall(request).execute()
                        val result = response.body?.string()
                        if (result != null && response.code == 200) {
                            if (result.length > 80) {
                                var tempResult: String? = result
                                while (tempResult != null && tempResult.length > 80) {
                                    Log.d(TAG, tempResult.substring(0, 79))
                                    tempResult = tempResult.drop(80)
                                }
                            }
                            val jsonDecoder = Json {
                                ignoreUnknownKeys = true
                                explicitNulls = false
                            }
                            val editedComponentList = jsonDecoder.decodeFromString<List<Component>>(result)
                            Log.d(
                                TAG,
                                "Received ${editedComponentList.size} edited components, updating..."
                            )
                            editedComponentList.forEach { editedComponent ->
                                if (editedComponent.uid > 0L) {
                                    val checkComponent =
                                        repository.getComponentByUid(editedComponent.uid)
                                    if (checkComponent != null) {
                                        repository.updateComponent(editedComponent)
                                        Log.d(
                                            TAG,
                                            "Updated ${editedComponent.uid}/${editedComponent.name}"
                                        )
                                    } else {
                                        repository.insertComponent(editedComponent)
                                        Log.d(
                                            TAG,
                                            "Inserted unknown ${editedComponent.uid}/${editedComponent.name}"
                                        )
                                    }
                                } else {
                                    repository.insertComponent(editedComponent)
                                    Log.d(
                                        TAG,
                                        "Inserted new/${editedComponent.name}"
                                    )
                                }
                            }
                            _reImportSuccessful.value = true
                        }
                    }
                }
            }
        } catch (exception: Exception) {
            Log.w(TAG, "Exception: $exception")
        }
    }

    private val _reImportSuccessful: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val reImportSuccessful: StateFlow<Boolean> = _reImportSuccessful
}