package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import android.util.Log
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.journeyapps.barcodescanner.ScanContract
import com.journeyapps.barcodescanner.ScanOptions
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ComponentEditDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.coroutines.launch

@Destination<RootGraph>
@Composable
fun ComponentRemoteEdit(
    componentRemoteEditViewModel: ComponentRemoteEditViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_edit_components_remotely))

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ComponentEditDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        headline = stringResource(R.string.hde_connect_with_component_manager), // TODO
        advancedFunctionality = advancedFunctionality
    ) {
        val codeIsValid by componentRemoteEditViewModel.codeIsValid.collectAsStateWithLifecycle()
        val uploadSuccessful by componentRemoteEditViewModel.uploadSuccessful.collectAsStateWithLifecycle()
        val reImportSuccessful by componentRemoteEditViewModel.reImportSuccessful.collectAsStateWithLifecycle()

        val scannedStringsSet: MutableSet<String> = mutableSetOf()

        val scope = rememberCoroutineScope()
        val scanLauncher = rememberLauncherForActivityResult(
            contract = ScanContract(),
            onResult = { result ->
                Log.i("CRE", "scanned code: ${result.contents}")
                val scannedString = result.contents
                if (!scannedStringsSet.contains(scannedString)) {
                    scannedStringsSet.add(scannedString)
                    scope.launch {
                        Log.d("CRE", "Read QR code $scannedString")
                        componentRemoteEditViewModel.setCode(
                            code = scannedString,
                            context = context
                        )
                    }
                    codeIsValid
                } else {
                    Log.d("CRE", "Seen code $scannedString before, will not action again.")
                }
            }
        )

        Column(
            modifier = Modifier.fillMaxWidth()
        ) {
            AnimatedVisibility(
                visible = !codeIsValid
            ) {
                Column {
                    Text(text = stringResource(R.string.text_scan_qr_code_on_jkbike_net_mdp_components))
                    DefaultSpacer()
                    FullWidthButton(onClick = {
                        scanLauncher.launch(ScanOptions())
                    }) {
                        Text(text = "Scan")
                    }
                }
            }
            AnimatedVisibility(
                visible = codeIsValid
            ) {
                Column {
                    Text(text = stringResource(R.string.text_scan_succesful))
                    DefaultSpacer()
                    AnimatedVisibility(
                        visible = !uploadSuccessful
                    ) {
                        Text(text = stringResource(R.string.text_uploading_components))
                    }
                    AnimatedVisibility(
                        visible = uploadSuccessful && !reImportSuccessful
                    ) {
                        Column {
                            Text(text = stringResource(R.string.text_you_can_now_edit_your_components_on_your_laptop_or_pc))
                            DefaultSpacer()
                            Text(text = stringResource(R.string.text_once_done_press_the_button_below_to_pull_your_edited_components_back_into_the_app))
                            DefaultSpacer()
                            FullWidthButton(
                                onClick = {
                                    componentRemoteEditViewModel.pullComponentsBack()
                                },
                                enabled = true
                            ) {
                                Text(text = stringResource(R.string.btn_retrieve_edited_components))
                            }
                        }
                    }
                    AnimatedVisibility(
                        reImportSuccessful
                    ) {
                        Text(text = stringResource(R.string.text_edited_components_have_been_updated_in_the_app))
                    }
                }
            }
        }
    }
}