package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.RetirementReason
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.WearLevel
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

@HiltViewModel(assistedFactory = ComponentEditViewModel.ComponentEditViewModelFactory::class)
class ComponentEditViewModel @AssistedInject constructor(
    @Assisted val componentUid: Long,
    val userPreferencesManager: UserPreferencesManager,
    val repository: KJsRepository
) : ViewModel() {

    private val _component: MutableStateFlow<Component?> = MutableStateFlow(null)
    val component: StateFlow<Component?> = _component

    val automaticallyPullDistanceFromBike = userPreferencesManager.automaticallyUpdateComponentDistanceFromBike().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        true
    )

    private val _currentMileageSuggestion: MutableStateFlow<Int?> = MutableStateFlow(null)
    val currentMileageSuggestion: StateFlow<Int?> = _currentMileageSuggestion

    val observeBikes = repository.observeBikes
    private val _currentBike: MutableStateFlow<Bike?> = MutableStateFlow(null)
    val currentBike: StateFlow<Bike?> = _currentBike

    private val _enablePeriodicChecks: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val enablePeriodicChecks: StateFlow<Boolean> = _enablePeriodicChecks

    init {
        if (componentUid > 0) {
            viewModelScope.launch {
                _component.value = repository.getComponentByUid(componentUid)
                // load the bike
                _currentBike.value = repository.getBikeByUid(component.value?.bikeUid ?: -1L)
                // update the currentMileage from mileageOffset + bike.mileage
                _currentMileageSuggestion.value = (component.value?.mileageOffset ?: 0) + (_currentBike.value?.mileage ?: 0)
            }
        }
    }

    fun updateName(newName: String) {
        if (component.value != null) {
            _component.value = component.value?.copy(
                name = newName
            )
        }
    }

    fun updateDescription(newDescription: String) {
        if (component.value != null) {
            _component.value = component.value?.copy(
                description = newDescription
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateAcquisitionDate(millis: Long?) {
        if (component.value != null) {
            if (millis != null) {
                val acquisitionDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val acquisitionDate =
                    acquisitionDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _component.value = component.value!!.copy(
                    acquisitionDate = acquisitionDate
                )
            } else {
                _component.value = component.value!!.copy(
                    acquisitionDate = null
                )
            }
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateLastCheckDate(millis: Long?) {
        if (component.value != null) {
            if (millis != null) {
                val lastCheckDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val lastCheckDate =
                    lastCheckDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _component.value = component.value!!.copy(
                    lastCheckDate = lastCheckDate
                )
            } else {
                _component.value = component.value!!.copy(
                    lastCheckDate = null
                )
            }
        }
    }

    fun updateLastCheckMileage(mileage: Int) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                lastCheckMileage = mileage
            )
        }
    }

    fun updateCurrentMileage(mileage: Int) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                currentMileage = mileage
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateFirstUsedDate(millis: Long?) {
        if (component.value != null) {
            if (millis != null) {
                val firstUsedDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val firstUsedDate =
                    firstUsedDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _component.value = component.value!!.copy(
                    firstUseDate = firstUsedDate
                )
            } else {
                _component.value = component.value!!.copy(
                    firstUseDate = null
                )
            }
        }
    }

    fun updateBike(bikeUid: Long?) {
        Log.d("ComponentEditVM", "Update bike $bikeUid...")
        if (bikeUid != null) {
            viewModelScope.launch {
                _currentBike.value = repository.getBikeByUid(bikeUid)
                Log.d("ComponentEditVM", "New bike: ${currentBike.value}")
                // re-calc mileageOffset
                if (component.value != null) {
                    val offset = (component.value!!.currentMileage ?: 0) - (currentBike.value?.mileage ?: 0)
                    _component.value = component.value!!.copy(
                        bikeUid = _currentBike.value?.uid,
                        mileageOffset = offset
                    )
                }
            }
        } else {
            _currentBike.value = null
        }
    }

    fun updateEnablePeriodicChecks(isEnabled: Boolean) {
        _enablePeriodicChecks.value = isEnabled
    }

    fun updateTitleForAutomaticActivities(title: String) {
        if (component.value != null) {
            _component.value = component.value?.copy(
                titleForAutomaticActivities = title
            )
        }
    }

    fun updateWearLevel(wearLevel: WearLevel?) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                wearLevel = wearLevel
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateRetirementDate(millis: Long?) {
        if (component.value != null) {
            if (millis != null) {
                val retirementDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val retirementDate =
                    retirementDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _component.value = component.value!!.copy(
                    retirementDate = retirementDate
                )
            } else {
                _component.value = component.value!!.copy(
                    retirementDate = null
                )
            }
        }
    }

    fun updateRetirementReason(reason: RetirementReason?) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                retirementReason = reason
            )
        }
    }

    fun updateCheckIntervalMileage(mileage: Int) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                checkIntervalMiles = mileage
            )
        }
    }

    fun updateCheckIntervalDays(days: Int) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                checkIntervalDays = days
            )
        }
    }

    fun updateNotifyWhenActivityCreated(enabled: Boolean) {
        if (component.value != null) {
            _component.value = component.value!!.copy(
                notifyWhenActivityCreated = enabled
            )
        }
    }

    fun updateAutomaticallyPullDistanceFromBike(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setAutomaticallyUpdateComponentDistanceFromBike(enabled)
        }
    }

    fun commitComponent() {
        if (component.value != null) {
            viewModelScope.launch {
                repository.updateComponent(component.value!!)
            }
        }
    }

    @AssistedFactory
    interface ComponentEditViewModelFactory {
        fun create(componentUid: Long): ComponentEditViewModel
    }
}