package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Done
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultBikeSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorNullableWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultNumberFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.RetirementReasonSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.WearLevelSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ComponentDeleteDestination
import com.ramcosta.composedestinations.generated.destinations.ComponentEditDestination
import com.ramcosta.composedestinations.generated.destinations.ManageComponentsDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.LocalTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun ComponentEdit(
    componentUid: Long,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_edit_a_component))

    val componentEditViewModel =
        hiltViewModel<ComponentEditViewModel, ComponentEditViewModel.ComponentEditViewModelFactory> { factory ->
            factory.create(componentUid = componentUid)
        }

    var modified by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ComponentEditDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            ),
            KJsAction(
                imageVector = Icons.Default.Delete,
                contentDescription = stringResource(R.string.btn_text_delete),
                onClick = {
                    destinationsNavigator.navigate(
                        ComponentDeleteDestination(componentUid)
                    )
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Done,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                componentEditViewModel.commitComponent()
                modified = false
                destinationsNavigator.popBackStack(
                    ManageComponentsDestination, inclusive = false
                )
            },
            enabled = modified
        ),
        headline = stringResource(R.string.hdr_edit_a_component),
        advancedFunctionality = advancedFunctionality
    ) {
        val component by componentEditViewModel.component.collectAsStateWithLifecycle()

        val acquisitionDateInstant = component?.let {
            it.acquisitionDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(
                    TimeZone.currentSystemDefault()
                )
            }
        }
        var selectedAcquisitionDate = acquisitionDateInstant?.toEpochMilliseconds()
        val firstUseDateInstant = component?.let {
            it.firstUseDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(TimeZone.currentSystemDefault())
            }
        }
        var selectedFirstUseDate = firstUseDateInstant?.toEpochMilliseconds()
        val lastCheckDateInstant = component?.let {
            it.lastCheckDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(TimeZone.currentSystemDefault())
            }
        }
        var selectedLastCheckDate = lastCheckDateInstant?.toEpochMilliseconds()
        val retirementDateInstant = component?.let {
            it.retirementDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(TimeZone.currentSystemDefault())
            }
        }
        var selectedRetirementDate = retirementDateInstant?.toEpochMilliseconds()

        val bikes: List<Bike> by componentEditViewModel.observeBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val currentBike: Bike? by componentEditViewModel.currentBike.collectAsStateWithLifecycle(
            initialValue = null
        )

        val currentMileageSuggestion by componentEditViewModel.currentMileageSuggestion.collectAsStateWithLifecycle()
        val automaticallyPullDistanceFromBike by componentEditViewModel.automaticallyPullDistanceFromBike.collectAsStateWithLifecycle()

        val stravaEnabled by kJsGlobalScaffoldViewModel.stravaEnabled.collectAsStateWithLifecycle()
        val intervalsEnabled by kJsGlobalScaffoldViewModel.intervalsEnabled.collectAsStateWithLifecycle()
        val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

        val enablePeriodicChecks by componentEditViewModel.enablePeriodicChecks.collectAsStateWithLifecycle()

        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            DefaultTextFieldWithSpacer(
                value = component?.name ?: stringResource(R.string.placehldr_name),
                label = stringResource(R.string.lbl_component_name),
                onValueChange = {
                    componentEditViewModel.updateName(it)
                    modified = true
                }
            )
            DefaultTextFieldWithSpacer(
                value = component?.description
                    ?: stringResource(R.string.placehldr_description),
                label = stringResource(R.string.lbl_description),
                onValueChange = {
                    componentEditViewModel.updateDescription(it)
                    modified = true
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedAcquisitionDate,
                label = stringResource(R.string.lbl_acquisition_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedAcquisitionDate = it
                    componentEditViewModel.updateAcquisitionDate(it)
                    modified = true
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedFirstUseDate,
                label = stringResource(R.string.lbl_first_use_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedFirstUseDate = it
                    componentEditViewModel.updateFirstUsedDate(it)
                    modified = true
                }
            )
            DefaultBikeSelectorWithSpacer(
                value = currentBike?.name ?: stringResource(R.string.dropdown_item_none),
                label = stringResource(R.string.lbl_attached_to_bike),
                bikes = bikes,
                onMenuItemClick = { bikeUid ->
                    componentEditViewModel.updateBike(bikeUid)
                    modified = true
                }
            )
            AnimatedVisibility(
                visible = currentBike != null
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.switch_automatically_update_distance_from_bike),
                    checked = automaticallyPullDistanceFromBike
                ) {
                    componentEditViewModel.updateAutomaticallyPullDistanceFromBike(it)
                    modified = true
                }
            }
            DefaultNumberFieldWithSpacer(
                value = (component?.currentMileage ?: "").toString(),
                label = stringResource(R.string.lbl_mileage),
                placeholder = (currentMileageSuggestion ?: "").toString(),
                enabled = automaticallyPullDistanceFromBike != true || currentBike == null,
                onValueChange = {
                    componentEditViewModel.updateCurrentMileage(it.toIntOrNull() ?: 0)
                    modified = true
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedLastCheckDate,
                label = stringResource(R.string.lbl_last_checked_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedLastCheckDate = it
                    componentEditViewModel.updateLastCheckDate(it)
                    modified = true
                }
            )
            DefaultNumberFieldWithSpacer(
                value = (component?.lastCheckMileage ?: "").toString(),
                label = stringResource(R.string.lbl_last_checked_mileage),
                onValueChange = {
                    componentEditViewModel.updateLastCheckMileage(it.toIntOrNull() ?: 0)
                    modified = true
                }
            )
            WearLevelSelector(
                currentWearLevel = component?.wearLevel,
                onWearLevelSelected = {
                    componentEditViewModel.updateWearLevel(it)
                    modified = true
                }
            )
            AnimatedVisibility(
                visible = stravaEnabled || intervalsEnabled
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.lbl_enable_periodic_checks),
                    checked = enablePeriodicChecks,
                    onCheckedChange = {
                        componentEditViewModel.updateEnablePeriodicChecks(it)
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.switch_send_notification_when_an_activity_is_created),
                    checked = component?.notifyWhenActivityCreated ?: false
                ) {
                    componentEditViewModel.updateNotifyWhenActivityCreated(it)
                }
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultNumberFieldWithSpacer(
                    value = (component?.checkIntervalMiles ?: "").toString(),
                    label = stringResource(R.string.lbl_check_interval_miles),
                    onValueChange = {
                        componentEditViewModel.updateCheckIntervalMileage(it.toIntOrNull() ?: 0)
                        modified = true
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultNumberFieldWithSpacer(
                    value = (component?.checkIntervalDays ?: "").toString(),
                    label = stringResource(R.string.lbl_check_interval_days),
                    onValueChange = {
                        componentEditViewModel.updateCheckIntervalDays(it.toIntOrNull() ?: 0)
                        modified = true
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultTextFieldWithSpacer(
                    value = component?.titleForAutomaticActivities ?: "",
                    label = stringResource(R.string.lbl_title_for_automatic_activities),
                    onValueChange = {
                        componentEditViewModel.updateTitleForAutomaticActivities(it)
                        modified = true
                    }
                )
            }
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedRetirementDate,
                label = stringResource(R.string.lbl_retirement_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedRetirementDate = it
                    componentEditViewModel.updateRetirementDate(it)
                    modified = true
                }
            )
            AnimatedVisibility(
                visible = selectedRetirementDate != null
            ) {
                RetirementReasonSelector(
                    currentRetirementReason = component?.retirementReason,
                    onRetirementReasonSelected = {
                        componentEditViewModel.updateRetirementReason(it)
                        modified = true
                    }
                )
            }
        }
    }
}