package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.datetime.daysUntil
import javax.inject.Inject

@HiltViewModel
class ComponentAnalysisViewModel @Inject constructor(
    userPreferencesManager: UserPreferencesManager,
    val repository: KJsRepository
) : ViewModel() {

    val retiredComponents = repository.observeRetiredComponents

    private val _listOfComponentUidsForAnalysis: MutableStateFlow<List<Long>> = MutableStateFlow(
        emptyList()
    )
    val listOfComponentUidsForAnalysis: StateFlow<List<Long>> = _listOfComponentUidsForAnalysis

    val distanceMeasure: StateFlow<DistanceMeasure> = userPreferencesManager.distanceMeasure().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        DistanceMeasure.KM
    )

    fun addUidToList(uid: Long) {
        val tempList = listOfComponentUidsForAnalysis.value.toMutableList()
        tempList.add(uid)
        _listOfComponentUidsForAnalysis.value = tempList
        runAnalysis()
    }

    fun removeUidFromList(uid: Long) {
        val tempList = listOfComponentUidsForAnalysis.value.toMutableList()
        tempList.remove(uid)
        _listOfComponentUidsForAnalysis.value = tempList
        runAnalysis()
    }

    private val _componentAnalysisResults: MutableStateFlow<ComponentAnalysisResults?> =
        MutableStateFlow(null)
    val componentAnalysisResults: StateFlow<ComponentAnalysisResults?> = _componentAnalysisResults

    fun runAnalysis() {
        var totalComponents = 0
        var totalMiles = 0
        var totalUsageDays = 0
        viewModelScope.launch {
            if (listOfComponentUidsForAnalysis.value.isNotEmpty()) {
                listOfComponentUidsForAnalysis.value.forEach { componentUid ->
                    totalComponents++
                    val component = repository.getComponentByUid(componentUid)
                    if (component != null) {
                        totalMiles += component.currentMileage ?: 0
                        if (component.retirementDate != null) {
                            if (component.firstUseDate != null) {
                                val lifetimeInDays =
                                    component.firstUseDate.daysUntil(component.retirementDate)
                                totalUsageDays += lifetimeInDays
                            } else if (component.acquisitionDate != null) {
                                val estimatedLifetimeInDays =
                                    component.acquisitionDate.daysUntil(component.retirementDate)
                                totalUsageDays += estimatedLifetimeInDays
                            }
                        }
                    }
                }
                val result = ComponentAnalysisResults(
                    numberOfComponents = totalComponents,
                    totalUsageMiles = totalMiles,
                    averageUsageMiles = (totalMiles / totalComponents),
                    totalUsageDays = totalUsageDays,
                    averageUsageDays = (totalUsageDays / totalComponents),
                    distanceMeasure = distanceMeasure.value
                )
                _componentAnalysisResults.value = result
            } else {
                _componentAnalysisResults.value = null
            }
        }
    }
}