package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import android.content.Context
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.Checkbox
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.material3.windowsizeclass.WindowWidthSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.toLocalisedString
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.RetiredComponents
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.ComponentAnalysisResultsTable
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ComponentAnalysisDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.daysUntil


@Destination<RootGraph>
@Composable
fun ComponentAnalysis(
    componentAnalysisViewModel: ComponentAnalysisViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_analyse_components))

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ComponentAnalysisDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        headline = stringResource(R.string.hdr_analyse_components),
        advancedFunctionality = advancedFunctionality
    ) {
        val retiredComponents: List<RetiredComponents> by componentAnalysisViewModel.retiredComponents.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val listOfComponentUidsForAnalysis by componentAnalysisViewModel.listOfComponentUidsForAnalysis.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        val analysisResults by componentAnalysisViewModel.componentAnalysisResults.collectAsStateWithLifecycle(
            initialValue = null
        )

        val distanceMeasure by componentAnalysisViewModel.distanceMeasure.collectAsStateWithLifecycle(
            initialValue = DistanceMeasure.KM
        )

        if (retiredComponents.isNotEmpty()) {
            if (windowSizeClass.widthSizeClass == WindowWidthSizeClass.Expanded) {
                Row(
                    modifier = Modifier.fillMaxSize()
                ) {
                    Box(
                        modifier = Modifier.weight(0.66f)
                    ) {
                        ListRetiredComponents(
                            retiredComponents,
                            listOfComponentUidsForAnalysis,
                            context,
                            distanceMeasure,
                            windowSizeClass.widthSizeClass,
                            onCheckedChange = { (checked, uuid) ->
                                if (checked) {
                                    componentAnalysisViewModel.addUidToList(uuid)
                                } else {
                                    componentAnalysisViewModel.removeUidFromList(
                                        uuid
                                    )
                                }
                            }
                        )
                    }
                    Column(
                        modifier = Modifier
                            .weight(0.33f)
                            .fillMaxHeight()
                    ) {
                        if (analysisResults != null) {
                            Text(
                                modifier = Modifier.fillMaxWidth(),
                                text = stringResource(R.string.text_analysis_of_components) + "(${analysisResults!!.numberOfComponents} )",
                                textAlign = TextAlign.Center
                            )
                            DefaultSpacer()
                            ComponentAnalysisResultsTable(analysisResults!!)
                        }
                    }
                }
            } else {
                Column(
                    modifier = Modifier.fillMaxSize()
                ) {
                    ListRetiredComponents(
                        retiredComponents,
                        listOfComponentUidsForAnalysis,
                        context,
                        distanceMeasure,
                        windowSizeClass.widthSizeClass,
                        onCheckedChange = { (checked, uuid) ->
                            if (checked) {
                                componentAnalysisViewModel.addUidToList(uuid)
                            } else {
                                componentAnalysisViewModel.removeUidFromList(
                                    uuid
                                )
                            }
                        }
                    )
                    DefaultSpacer()
                    if (analysisResults != null) {
                        Text(
                            modifier = Modifier.padding(8.dp, 0.dp),
                            text = stringResource(R.string.text_analysis_of_components) + "(${analysisResults!!.numberOfComponents} )"
                        )
                        DefaultSpacer()
                        ComponentAnalysisResultsTable(analysisResults!!)
                    }
                }
            }
        } else { // there are no retired components to analyse
            Text(text = stringResource(R.string.there_are_no_retired_components_to_analyse))
        }
    }
}

@OptIn(ExperimentalFoundationApi::class)
@Composable
private fun ListRetiredComponents(
    retiredComponents: List<RetiredComponents>,
    listOfComponentUidsForAnalysis: List<Long>,
    context: Context,
    distanceMeasure: DistanceMeasure,
    windowWidthSizeClass: WindowWidthSizeClass,
    onCheckedChange: (Pair<Boolean, Long>) -> Unit
) {
    LazyColumn {
        stickyHeader {
            Text(
                modifier = Modifier
                    .fillMaxWidth()
                    .background(MaterialTheme.colorScheme.primaryContainer)
                    .padding(8.dp),
                text = stringResource(R.string.hdr_components),
                style = MaterialTheme.typography.bodyMedium,
            )
        }

        items(retiredComponents, key = { it.uid }) { retiredComponent ->
            Row(
                modifier = Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Checkbox(
                    modifier = Modifier.weight(0.07f),
                    checked = listOfComponentUidsForAnalysis.contains(
                        retiredComponent.uid
                    ),
                    onCheckedChange = {
                        onCheckedChange(Pair(it, retiredComponent.uid))
                    }
                )
                Column(
                    modifier = Modifier.weight(0.93f)
                ) {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Text(
                            text = retiredComponent.name,
                            overflow = TextOverflow.Ellipsis,
                            modifier = Modifier.weight(0.3f)
                        )
                        if (windowWidthSizeClass != WindowWidthSizeClass.Compact) {
                            Text(
                                text = retiredComponent.currentMileage.toString() + " " + distanceMeasure.metric,
                                modifier = Modifier.weight(0.2f)
                            )
                            if (retiredComponent.retirementDate != null) {
                                val daysInUse = if (retiredComponent.firstUseDate != null) {
                                    retiredComponent.firstUseDate.daysUntil(retiredComponent.retirementDate)
                                } else if (retiredComponent.acquisitionDate != null) {
                                    retiredComponent.acquisitionDate.daysUntil(
                                        retiredComponent.retirementDate
                                    )
                                } else {
                                    0
                                }
                                Text(
                                    text = "$daysInUse" + context.getString(R.string.lbl_days),
                                    modifier = Modifier.weight(0.2f)
                                )
                            } else {
                                Text(text = "", modifier = Modifier.weight(0.2f))
                            }
                        }
                        Text(
                            text = if (retiredComponent.retirementReason != null) {
                                toLocalisedString(
                                    retiredComponent.retirementReason,
                                    context
                                )
                            } else {
                                ""
                            },
                            modifier = Modifier.weight(0.3f),
                            overflow = TextOverflow.Ellipsis,
                            maxLines = 1
                        )
                    }
                    AnimatedVisibility(
                        visible = retiredComponent.description.isNotEmpty() || windowWidthSizeClass != WindowWidthSizeClass.Compact
                    ) {
                        Row {
                            val labelColor = MaterialTheme.colorScheme.secondary
                            Text(
                                text = retiredComponent.description,
                                style = MaterialTheme.typography.labelMedium,
                                color = labelColor,
                                modifier = Modifier.weight(0.5f)
                            )
                            if (windowWidthSizeClass != WindowWidthSizeClass.Compact) {
                                val timeText = if (retiredComponent.firstUseDate != null) {
                                    retiredComponent.firstUseDate.toString()
                                } else if (retiredComponent.acquisitionDate != null) {
                                    retiredComponent.acquisitionDate.toString()
                                } else {
                                    "?"
                                } + " > " + if (retiredComponent.retirementDate != null) {
                                    retiredComponent.retirementDate.toString()
                                } else {
                                    "?"
                                }
                                Text(
                                    text = timeText,
                                    style = MaterialTheme.typography.labelMedium,
                                    color = labelColor,
                                    modifier = Modifier.weight(.5f)
                                )
                            }
                        }
                    }
                }
            }
        }
    }
}