package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.Save
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.RetirementReason
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.WearLevel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.BikeSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorNullableWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultNumberFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.RetirementReasonSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.WearLevelSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ComponentAddDestination
import com.ramcosta.composedestinations.generated.destinations.ManageComponentsDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun ComponentAdd(
    componentAddViewModel: ComponentAddViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_add_component))

    // input fields
    var name by remember { mutableStateOf("") }
    var description by remember { mutableStateOf("") }
    var acquisitionDate by remember { mutableStateOf<Long?>(null) }
    var firstUseDate by remember { mutableStateOf<Long?>(null) }
    var lastCheckDate by remember { mutableStateOf<Long?>(null) }
    var bike: Bike? by remember { mutableStateOf(null) }
    var checkIntervalMiles: Int? by remember { mutableStateOf(null) }
    var checkIntervalDays: Int? by remember { mutableStateOf(null) }
    var lastCheckMileage: Int? by remember { mutableStateOf(null) }
    var currentMileage: Int? by remember { mutableStateOf(null) }
    var mileageOffset: Int? by remember { mutableStateOf(null) }
    var titleForAutomaticActivities: String? by remember { mutableStateOf(null) }
    var wearLevel: WearLevel? by remember { mutableStateOf(null) }
    var retirementDate by remember { mutableStateOf<Long?>(null) }
    var retirementReason: RetirementReason? by remember { mutableStateOf(null) }
    var notifyWhenActivityCreated by remember { mutableStateOf(false) }

    val allBikes by componentAddViewModel.allBikes.collectAsStateWithLifecycle(emptyList())

    val stravaEnabled by kJsGlobalScaffoldViewModel.stravaEnabled.collectAsStateWithLifecycle()
    val intervalsEnabled by kJsGlobalScaffoldViewModel.intervalsEnabled.collectAsStateWithLifecycle()
    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    var enablePeriodicChecks by remember { mutableStateOf(false) }

    var modified by remember { mutableStateOf(false) }

    KJsResponsiveNavigation(
        ComponentAddDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.Default.Clear,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Save,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                val component = Component(
                    name = name,
                    description = description,
                    acquisitionDate = if (acquisitionDate != null) {
                        Instant.fromEpochMilliseconds(acquisitionDate!!)
                            .toLocalDateTime(
                                TimeZone.currentSystemDefault()
                            ).date
                    } else {
                        null
                    },
                    firstUseDate = if (firstUseDate != null) {
                        Instant.fromEpochMilliseconds(firstUseDate!!)
                            .toLocalDateTime(
                                TimeZone.currentSystemDefault()
                            ).date
                    } else {
                        null
                    },
                    lastCheckDate = if (lastCheckDate != null) {
                        Instant.fromEpochMilliseconds(lastCheckDate!!)
                            .toLocalDateTime(
                                TimeZone.currentSystemDefault()
                            ).date
                    } else {
                        null
                    },
                    bikeUid = bike?.uid,
                    checkIntervalMiles = checkIntervalMiles,
                    checkIntervalDays = checkIntervalDays,
                    lastCheckMileage = lastCheckMileage,
                    currentMileage = currentMileage,
                    mileageOffset = mileageOffset,
                    titleForAutomaticActivities = titleForAutomaticActivities,
                    wearLevel = wearLevel,
                    retirementDate = if (retirementDate != null) {
                        Instant.fromEpochMilliseconds(retirementDate!!)
                            .toLocalDateTime(
                                TimeZone.currentSystemDefault()
                            ).date
                    } else {
                        null
                    },
                    retirementReason = retirementReason,
                    notifyWhenActivityCreated = notifyWhenActivityCreated,
                    lastCheckNotificationMileage = null,
                    lastCheckNotificationDate = null,
                    uid = 0L
                )
                componentAddViewModel.saveNewComponent(
                    component = component,
                )
                modified = false
                destinationsNavigator.navigateUp()
                destinationsNavigator.popBackStack(
                    ManageComponentsDestination, inclusive = false
                )
            },
            enabled = modified && name.isNotBlank() && acquisitionDate != null
        ),
        headline = stringResource(R.string.hdr_add_component),
        advancedFunctionality = advancedFunctionality
    ) {
        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            DefaultTextFieldWithSpacer(
                value = name,
                onValueChange = {
                    name = it
                    modified = true
                },
                label = stringResource(R.string.lbl_name),
            )
            DefaultTextFieldWithSpacer(
                value = description,
                onValueChange = {
                    description = it
                    modified = true
                },
                label = stringResource(R.string.lbl_description),
            )
            DefaultDateSelectorWithSpacer(
                selectedDate = acquisitionDate,
                label = stringResource(R.string.lbl_acquisition_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    acquisitionDate = it
                    modified = true
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = firstUseDate,
                label = stringResource(R.string.lbl_first_use_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    firstUseDate = it
                    modified = true
                }
            )
            // bike selector here
            BikeSelector(
                currentBike = bike,
                bikes = allBikes,
                label = stringResource(R.string.dropdown_lbl_which_bike)
            ) { selectedBike ->
                bike = selectedBike
                // re-calc mileageOffset
                mileageOffset = (currentMileage ?: 0) - (bike?.mileage ?: 0)
                modified = true
            }
            DefaultSpacer()
            DefaultNumberFieldWithSpacer(
                value = (currentMileage ?: "").toString(),
                label = stringResource(R.string.lbl_current_mileage),
                placeholder = "0",
                onValueChange = {
                    currentMileage = it.toIntOrNull() ?: currentMileage
                    modified = true
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = lastCheckDate,
                label = stringResource(R.string.lbl_last_checked_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    lastCheckDate = it
                    modified = true
                }
            )
            DefaultNumberFieldWithSpacer(
                value = (lastCheckMileage ?: "").toString(),
                label = stringResource(R.string.lbl_last_checked_mileage),
                placeholder = "0",
                onValueChange = {
                    lastCheckMileage = it.toIntOrNull() ?: lastCheckMileage
                    modified = true
                }
            )
            // wear level selector here
            WearLevelSelector(
                currentWearLevel = wearLevel,
                onWearLevelSelected = {
                    wearLevel = it
                    modified = true
                }
            )
            AnimatedVisibility(
                visible = stravaEnabled || intervalsEnabled
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.lbl_enable_periodic_checks),
                    checked = enablePeriodicChecks,
                    onCheckedChange = {
                        enablePeriodicChecks = it
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.switch_send_notification_when_an_activity_is_created),
                    checked = notifyWhenActivityCreated
                ) {
                    notifyWhenActivityCreated = it
                }
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultNumberFieldWithSpacer(
                    value = (checkIntervalMiles ?: "").toString(),
                    label = stringResource(R.string.lbl_check_interval_miles),
                    placeholder = "0",
                    onValueChange = {
                        checkIntervalMiles = it.toIntOrNull() ?: checkIntervalMiles
                        modified = true
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultNumberFieldWithSpacer(
                    value = (checkIntervalDays ?: "").toString(),
                    label = stringResource(R.string.lbl_check_interval_days),
                    placeholder = "0",
                    onValueChange = {
                        checkIntervalDays = it.toIntOrNull() ?: checkIntervalDays
                        modified = true
                    }
                )
            }
            AnimatedVisibility(
                visible = enablePeriodicChecks && (stravaEnabled || intervalsEnabled)
            ) {
                DefaultTextFieldWithSpacer(
                    value = titleForAutomaticActivities ?: "",
                    label = stringResource(R.string.lbl_title_for_automatic_activities),
                    onValueChange = {
                        titleForAutomaticActivities = it
                        modified = true
                    }
                )
            }
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = retirementDate,
                label = stringResource(R.string.lbl_retirement_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    retirementDate = it
                    modified = true
                }
            )
            AnimatedVisibility(
                visible = retirementDate != null
            ) {
                RetirementReasonSelector(
                    currentRetirementReason = retirementReason,
                    onRetirementReasonSelected = {
                        retirementReason = it
                        modified = true
                    }
                )
            }
        }
    }
}