package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.checklist

import android.content.Context
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.PopulateDatabaseHelpersViaRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.loadTemplateActivitiesJsonAndImport
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.SupportedLanguages
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.todayIn
import javax.inject.Inject
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

private const val TAG = "PfRVM"

@HiltViewModel
class PrepareForRideViewModel @Inject constructor(
    private val repository: KJsRepository
) : ViewModel() {

    val observeBikes = repository.observeBikes

    val observeTemplateActivitys = repository.observeTemplateActivity

    init {
        viewModelScope.launch {
            // check that we have templates!
            // try loading them, if necessary, and fall back to creating them in English
            val numberOfTemplates = observeTemplateActivitys.firstOrNull()?.size ?: 0
            if (numberOfTemplates == 0) {
                // we must load!
                val languageCode = SupportedLanguages.determineLocaleToLoad()
                Log.d(TAG, "Will try to load templates for $languageCode...")
                loadTemplateActivitiesJsonAndImport(
                    language = languageCode,
                    importDataCallback = { templateActivities ->
                        if (templateActivities.isNotEmpty()) {
                            Log.d(TAG, "Found ${templateActivities.size} templates, adding...")
                            CoroutineScope(Dispatchers.IO).launch {
                                // clear out old templates
                                repository.deleteAllBuiltInTemplateActivities()
                                // import new list
                                templateActivities.forEach { templateActivity ->
                                    repository.insertTemplateActivity(templateActivity)
                                }
                                Log.d(TAG, "Import done.")
                            }
                        }
                    },
                    errorCallback = {
                        Log.w(TAG, "Something wrong, will add static en templates...")
                        CoroutineScope(Dispatchers.IO).launch {
                            val creator = PopulateDatabaseHelpersViaRepository(repository)
                            creator.createFallbackTemplateActivities()
                        }
                    }
                )
            }
        }
    }

    @OptIn(ExperimentalTime::class)
    fun createRideActivities(
        rideLevel: Int,
        bike: Bike?,
        isEBike: Boolean?,
        context: Context,
        successAction: (Long) -> Unit
    ) {
        viewModelScope.launch {
            // 0 - delete all historic rides
//             repository.deleteAllRides() // TODO

            // 1 - create a ride
            val ride = Ride(
                name = "${
                    RideLevel.getLabel(
                        context,
                        rideLevel
                    )
                } - ${Clock.System.todayIn(TimeZone.currentSystemDefault())}",
                level = rideLevel,
                createdInstant = Clock.System.now(),
                completed = false,
                completedInstant = null,
                bikeUid = bike?.uid
            )
            Log.d(TAG, "Created new ride $ride")
            val rideUid = repository.insertRide(ride)

            // 2 - add all activities from template
            val templateActivities = repository.getTemplateActivitiesForRideLevel(rideLevel)
            Log.d(TAG, "${templateActivities.size} templates found, filtering...")
            templateActivities.filter { template ->
                !template.isEBikeSpecific || isEBike == true
            }.forEach { template ->
                val activity = Activity(
                    title = template.title,
                    description = template.description,
                    isCompleted = false,
                    bikeUid = bike?.uid,
                    isEBikeSpecific = template.isEBikeSpecific,
                    rideLevel = template.rideLevel,
                    componentUid = null,
                    rideUid = rideUid,
                    createdInstant = ride.createdInstant,
                    dueDate = null,
                    doneInstant = null,
                    uid = 0L
                )
                Log.d(TAG, "Creating $activity")
                repository.insertActivity(activity)
            }

            // 3 - done
            successAction(rideUid)
        }
    }

    companion object {
        private const val TAG = "PrepareForRideVM"
    }
}