package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.checklist

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ThumbUp
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.style.LineBreak
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.BikeSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ChecklistDestination
import com.ramcosta.composedestinations.generated.destinations.PrepareForRideDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@Destination<RootGraph>
@Composable
fun PrepareForRide(
    prepareForRideViewModel: PrepareForRideViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_prepare_for_a_ride))

    val context = LocalContext.current

    var currentRideLevel by remember { mutableIntStateOf(RideLevel.getRideLevelQuickRide()) }

    var currentBike: Bike? by remember { mutableStateOf(null) }

    var currentBikeIsAnEbike: Boolean? by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        PrepareForRideDestination,
        destinationsNavigator,
        windowSizeClass,
        myFloatingActionButton = KJsAction(
            enabled = true,
            imageVector = Icons.Default.ThumbUp,
            contentDescription = stringResource(R.string.btn_text_create),
            onClick = {
                prepareForRideViewModel.createRideActivities(
                    rideLevel = currentRideLevel,
                    bike = currentBike,
                    isEBike = currentBikeIsAnEbike,
                    context = context,
                    successAction = { rideUid ->
                        destinationsNavigator.navigate(ChecklistDestination(rideUid))
                    }
                )
            }
        ),
        headline = "",
        advancedFunctionality = advancedFunctionality
    ) {
        val bikes by prepareForRideViewModel.observeBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        Column(
            modifier = Modifier
                .fillMaxWidth()
                .padding(8.dp)
                .verticalScroll(rememberScrollState())
        ) {
            Spacer(modifier = Modifier.weight(0.5f))

            // 1 - what kind of ride is it?
            Text(
                text = stringResource(R.string.lbl_type_of_ride)
            )
            DefaultSpacer()
            Column(
                modifier = Modifier.padding(0.dp, 0.dp, 8.dp, 0.dp)
            ) {
                RideLevel.getListOfRideLevels().forEach { thisRideLevel ->
                    Column(
                        modifier = Modifier
                            .clip(RoundedCornerShape(8.dp))
                            .clickable {
                                currentRideLevel = thisRideLevel
                            }
                            .background(
                                if (thisRideLevel == currentRideLevel) {
                                    MaterialTheme.colorScheme.primary
                                } else {
                                    MaterialTheme.colorScheme.surface
                                }
                            )
                            .padding(
                                vertical = 4.dp,
                                horizontal = 8.dp,
                            ),
                    ) {
                        Text(
                            text = RideLevel.getHeading(
                                context = context,
                                level = thisRideLevel
                            ),
                            color = if (thisRideLevel == currentRideLevel) {
                                MaterialTheme.colorScheme.onPrimary
                            } else {
                                MaterialTheme.colorScheme.onSurface
                            },
                            style = MaterialTheme.typography.titleMedium
                        )
                        Spacer(modifier = Modifier.size(3.dp))
                        Text(
                            text = RideLevel.getTimeframe(
                                context = context,
                                level = thisRideLevel
                            ),
                            style = MaterialTheme.typography.bodyMedium,
                            fontStyle = FontStyle.Italic,
                            color = if (thisRideLevel == currentRideLevel) {
                                MaterialTheme.colorScheme.onPrimary
                            } else {
                                MaterialTheme.colorScheme.onSurface
                            },
                        )
                        Spacer(modifier = Modifier.size(4.dp))
                        Text(
                            text = RideLevel.getDescription(
                                context = context,
                                level = thisRideLevel
                            ),
                            color = if (thisRideLevel == currentRideLevel) {
                                MaterialTheme.colorScheme.onPrimary
                            } else {
                                MaterialTheme.colorScheme.onSurface
                            },
                            style = MaterialTheme.typography.bodySmall.copy(
                                lineBreak = LineBreak.Paragraph
                            )
                        )
                    }
                    DefaultSpacer()
                }
            }

            DefaultSpacer()

            // 2 - any specific bike?
            if (bikes.isNotEmpty()) {
                BikeSelector(
                    modifier = Modifier.fillMaxWidth(),
                    currentBike = currentBike,
                    bikes = bikes,
                    label = stringResource(R.string.lbl_which_bike_will_you_ride)
                ) {
                    currentBike = it
                    currentBikeIsAnEbike = it?.isElectric == true
                }
            } else {
                TextAndSwitch(
                    text = stringResource(R.string.lbl_my_bike_is_an_ebike),
                    checked = currentBikeIsAnEbike == true,
                ) {
                    currentBikeIsAnEbike = it
                }
            }
        }
    }
}