package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.checklist

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.WearLevel
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.todayIn
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@HiltViewModel(assistedFactory = ChecklistViewModel.ChecklistViewModelFactory::class)
class ChecklistViewModel @AssistedInject constructor(
    @Assisted val rideUid: Long,
    private val repository: KJsRepository
) : ViewModel() {

    private val _ride: MutableStateFlow<Ride?> = MutableStateFlow(null)
    val ride: StateFlow<Ride?> = _ride

    val observeActivityWithBikeData = repository.observeActivityWithBikeAndRideDataOrderedByDueDate

    private val componentsLookupTable: MutableMap<Long, Component> = mutableMapOf()

    init {
        if (rideUid > 0) {
            viewModelScope.launch {
                _ride.value = repository.getRideByUid(rideUid)
                val components = repository.getAllComponents()
                components.forEach { component ->
                    componentsLookupTable.put(component.uid, component)
                }
            }
        }
    }

    fun getComponentNameForUid(componentUid: Long?): String? {
        if (componentUid == null) {
            return null
        }
        return componentsLookupTable[componentUid]?.name
    }

    fun finishRide() {
        // mark ride complete
        viewModelScope.launch {
            val tempRide = ride.value
            if (tempRide != null) {
                val now = Clock.System.now()
                val closedRide = tempRide.copy(
                    completed = true,
                    completedInstant = now
                )
                repository.updateRide(closedRide)
            }
        }
    }

    fun updateActivity(activity: Activity) {
        viewModelScope.launch {
            repository.updateActivity(activity)
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateAttachedComponent(
        componentUid: Long,
        bikeUid: Long?
    ) {
        viewModelScope.launch {
            val bikeMileage = if (bikeUid != null) {
                repository.getBikeByUid(bikeUid)?.mileage ?: 0
            } else {
                0
            }
            val component = repository.getComponentByUid(componentUid)?.copy(
                lastCheckDate = Clock.System.todayIn(TimeZone.Companion.currentSystemDefault()),
            )
            val newComponent = if (bikeMileage > 0) {
                component?.copy(
                    lastCheckMileage = bikeMileage - (component.mileageOffset ?: 0)
                )
            } else {
                component
            }
            if (newComponent != null) {
                repository.updateComponent(newComponent)
            }
        }
    }

    fun logComponentWearLevel(componentUid: Long, wearLevel: WearLevel) {
        viewModelScope.launch {
            val component = repository.getComponentByUid(componentUid)
            if (component != null) {
                val updatedComponent = component.copy(
                    wearLevel = wearLevel
                )
                repository.updateComponent(updatedComponent)
            }
        }
    }

    @AssistedFactory
    interface ChecklistViewModelFactory {
        fun create(rideUid: Long): ChecklistViewModel
    }
}
