package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.checklist

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Done
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowHeightSizeClass
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ActivityWithBikeAndRideData
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.ComponentWearLevelDialog
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TodoListItem
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ChecklistDestination
import com.ramcosta.composedestinations.generated.destinations.HomeDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalFoundationApi::class, ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun Checklist(
    rideUid: Long,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle("") // Home doesn't have a title

    val checklistViewModel =
        hiltViewModel<ChecklistViewModel, ChecklistViewModel.ChecklistViewModelFactory> { factory ->
            factory.create(rideUid = rideUid)
        }

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ChecklistDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                enabled = true,
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.popBackStack(HomeDestination, inclusive = false)
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            enabled = true,
            imageVector = Icons.Default.Done,
            contentDescription = stringResource(R.string.btn_desc_done_preparing),
            onClick = {
                checklistViewModel.finishRide()
                destinationsNavigator.popBackStack(HomeDestination, inclusive = false)
            }
        ),
        headline = "",
        advancedFunctionality = advancedFunctionality
    ) {
        val activitiesByBikes: List<ActivityWithBikeAndRideData> by checklistViewModel
            .observeActivityWithBikeData.collectAsStateWithLifecycle(
                initialValue = emptyList()
            )
        val relevantOpenActivitiesByBikes = activitiesByBikes.filter { activityByBike ->
            activityByBike.rideIsCompleted != true && activityByBike.rideUid == rideUid
        }
        val ride by checklistViewModel.ride.collectAsStateWithLifecycle()
        var showDialog by remember { mutableStateOf(false) }

        Column(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp)
        ) {
            if (relevantOpenActivitiesByBikes.isNotEmpty()) {
                LazyColumn(
                    modifier = Modifier
                        .fillMaxWidth()
                ) {
                    stickyHeader {
                        if (windowSizeClass.heightSizeClass != WindowHeightSizeClass.Compact) {
                            Text(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .background(MaterialTheme.colorScheme.primaryContainer)
                                    .padding(8.dp),
                                text = "${stringResource(R.string.hdr_todos)} - ${RideLevel.getLabel(
                                    context = context,
                                    level = ride?.level)}"
                            )
                        }
                    }

                    items(
                        items = relevantOpenActivitiesByBikes,
                        key = { it.activityUid }) { activityByBike ->
                        val activity = Activity(
                            title = activityByBike.activityTitle,
                            description = activityByBike.activityDescription,
                            isCompleted = activityByBike.activityIsCompleted,
                            rideUid = activityByBike.rideUid,
                            isEBikeSpecific = activityByBike.isEBikeSpecific,
                            rideLevel = activityByBike.activityRideLevel,
                            createdInstant = activityByBike.activityCreatedInstant,
                            dueDate = activityByBike.activityDueDate,
                            bikeUid = activityByBike.bikeUid ?: 0,
                            doneInstant = activityByBike.activityDoneDateInstant,
                            componentUid = activityByBike.activityComponentUid,
                            uid = activityByBike.activityUid
                        )
                        TodoListItem(
                            activity = activityByBike,
                            destinationsNavigator = destinationsNavigator,
                            onCheckboxCallback = { result ->
                                checklistViewModel.updateActivity(
                                    activity = activity.copy(
                                        isCompleted = result,
                                        doneInstant = if (result) {
                                            Clock.System.now()
                                        } else {
                                            null
                                        }
                                    )
                                )
                                // popup in case this has a component attached
                                if (activityByBike.activityComponentUid != null) {
                                    // update component
                                    checklistViewModel.updateAttachedComponent(
                                        componentUid = activityByBike.activityComponentUid,
                                        bikeUid = activityByBike.bikeUid
                                    )
                                    if (result) {
                                        // show UI for component wearlevel
                                        showDialog = true
                                    }
                                }
                            },
                        )
                        if (showDialog) {
                            ComponentWearLevelDialog(
                                onDismissRequest = {
                                    showDialog = false
                                },
                                onConfirmation = { wearLevel ->
                                    showDialog = false
                                    // write stuff
                                    if (activityByBike.activityComponentUid != null) {
                                        checklistViewModel.logComponentWearLevel(
                                            componentUid = activityByBike.activityComponentUid,
                                            wearLevel = wearLevel
                                        )
                                    }
                                },
                                componentName = checklistViewModel.getComponentNameForUid(
                                    activityByBike.activityComponentUid
                                )
                            )
                        }
                    }
                }
            } else {
                // no activities
                // TODO
            }
        }
    }
}
