package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.bikes

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.automirrored.filled.DirectionsBike
import androidx.compose.material.icons.filled.Add
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthOutlinedButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.IconSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.IntervalsAttributionTextAndLogo
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithIntervalsViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithStravaViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.BikeAddDestination
import com.ramcosta.composedestinations.generated.destinations.BikeEditDestination
import com.ramcosta.composedestinations.generated.destinations.ManageBikesDestination
import com.ramcosta.composedestinations.generated.destinations.SyncWithIntervalsDestination
import com.ramcosta.composedestinations.generated.destinations.SyncWithStravaDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@OptIn(ExperimentalFoundationApi::class)
@Destination<RootGraph>
@Composable
fun ManageBikes(
    manageBikesViewModel: ManageBikesViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    syncWithStravaViewModel: SyncWithStravaViewModel = hiltViewModel(),
    syncWithIntervalsViewModel: SyncWithIntervalsViewModel = hiltViewModel(),
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_manage_bikes))

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        currentDestination = ManageBikesDestination,
        destinationsNavigator = destinationsNavigator,
        windowSizeClass = windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Add,
            contentDescription = stringResource(R.string.btn_text_add_bike),
            onClick = {
                destinationsNavigator.navigate(BikeAddDestination)
            }
        ),
        headline = stringResource(R.string.hdr_manage_bikes),
        advancedFunctionality = advancedFunctionality
    ) {
        val bikes: List<Bike> by manageBikesViewModel.bikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        val allBikes: List<Bike> by manageBikesViewModel.allBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        var showAllBikes by remember { mutableStateOf(false) }

        val distanceMeasure by manageBikesViewModel.distanceMeasure.collectAsStateWithLifecycle(
            initialValue = DistanceMeasure.KM
        )

        val stravaEnabled by manageBikesViewModel.stravaEnabled.collectAsStateWithLifecycle()
        val isStravaConnected by syncWithStravaViewModel.isStravaConnected.collectAsStateWithLifecycle()
        val isAllowedToReadAllProfileData by syncWithStravaViewModel.isAllowedToReadAllProfileData.collectAsStateWithLifecycle()

        val intervalsEnabled by manageBikesViewModel.intervalsEnabled.collectAsStateWithLifecycle()
        val isIntervalsConnected by syncWithIntervalsViewModel.isIntervalsConnected.collectAsStateWithLifecycle()
        val isAllowedToReadAthleteSettings by syncWithIntervalsViewModel.isAllowedToReadAthleteSettings.collectAsStateWithLifecycle()

        Column(
            modifier = Modifier
                .fillMaxWidth()
        ) {

            TextAndSwitch(
                text = stringResource(R.string.show_inactive_bikes),
                checked = showAllBikes,
                onCheckedChange = {
                    showAllBikes = it
                }
            )

            if (stravaEnabled) {
                Text(
                    modifier = Modifier
                        .padding(8.dp)
                        .align(Alignment.End),
                    text = buildAnnotatedString {
                        withLink(
                            LinkAnnotation.Url(
                                url = "https://www.strava.com/settings/gear",
                                styles = TextLinkStyles(
                                    style = SpanStyle(color = MaterialTheme.colorScheme.primary)
                                )
                            )
                        ) {
                            append("View on Strava")
                        }
                    }
                )
                DefaultSpacer()
            }

            Text(
                modifier = Modifier
                    .fillMaxWidth()
                    .background(MaterialTheme.colorScheme.primaryContainer)
                    .padding(8.dp),
                text = stringResource(R.string.hdr_bikes),
                style = MaterialTheme.typography.bodyMedium,
            )

            val bikesToShow = if (showAllBikes) {
                allBikes
            } else {
                bikes
            }
            bikesToShow.forEach { bike ->
                Surface(
                    modifier = Modifier
                        .combinedClickable(
                            onClick = {
                                destinationsNavigator.navigate(BikeEditDestination(bike.uid))
                            },
                        )
                        .padding(4.dp),
                ) {
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                    ) {
                        val textColor = if (bike.isActive) {
                            Color.Unspecified
                        } else {
                            MaterialTheme.colorScheme.tertiary
                        }
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.DirectionsBike,
                            contentDescription = stringResource(R.string.bike),
                        )
                        IconSpacer()
                        Text(
                            text = bike.name,
                            modifier = Modifier.weight(1f),
                            color = textColor
                        )
                        DefaultSpacer()
                        Text(
                            text = "${bike.mileage} ${distanceMeasure.metric}",
                            color = textColor
                        )
                    }
                }
            }

            AnimatedVisibility(
                visible = bikesToShow.isEmpty()
            ) {
                if (stravaEnabled) {
                    Text(text = stringResource(R.string.text_no_bikes_so_far_you_can_add_a_bike_below_or_sync_with_strava))
                } else if (intervalsEnabled) {
                    Text(text = stringResource(R.string.text_no_bikes_so_far_you_can_add_a_bike_below_or_sync_with_intervals))
                } else {
                    Text(text = stringResource(R.string.text_no_bikes_so_far_you_can_add_a_bike_below))
                }
            }

            // STRAVA
            AnimatedVisibility(
                visible = stravaEnabled
            ) {
                Column {
                    DefaultSpacer()
                    Icon(
                        painter = painterResource(R.drawable.api_logo_pwrdby_strava_horiz_orange),
                        contentDescription = "Powered by Strava",
                        tint = Color.Unspecified,
                        modifier = Modifier
                            .height(24.dp)
                            .padding(8.dp, 0.dp)
                            .align(Alignment.End)
                    )
                    DefaultSpacer()
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.strava),
                        style = MaterialTheme.typography.bodyMedium
                    )
                    DefaultSpacer()
                    AnimatedVisibility(
                        visible = !isStravaConnected
                    ) {
                        val context = LocalContext.current
                        Icon(
                            painter = painterResource(R.drawable.btn_strava_connect_with_orange),
                            contentDescription = stringResource(R.string.connect_with_strava),
                            tint = Color.Unspecified,
                            modifier = Modifier
                                .align(Alignment.CenterHorizontally)
                                .clickable(
                                    enabled = stravaEnabled,
                                    onClick = {
                                        syncWithStravaViewModel.connect(context)
                                    }
                                )
                        )
                    }
                    AnimatedVisibility(
                        visible = isStravaConnected
                    ) {
                        Column {
                            FullWidthButton(
                                enabled = isAllowedToReadAllProfileData && bikesToShow.isNotEmpty(),
                                onClick = {
                                    syncWithStravaViewModel.updateDistancesFromStrava(
                                        context = context,
                                        onResult = {
                                            // nothing to do
                                        },
                                        onError = { code ->
                                            // TODO
                                        }
                                    )
                                }
                            ) {
                                Text(text = stringResource(R.string.btn_text_update_distances_from_strava))
                            }
                            DefaultSpacer()
                            FullWidthOutlinedButton(onClick = {
                                destinationsNavigator.navigate(SyncWithStravaDestination)
                            }) {
                                Text(text = stringResource(R.string.dropdown_item_sync_with_strava))
                            }
                            if (!isAllowedToReadAllProfileData) {
                                DefaultSpacer()
                                Text(text = stringResource(R.string.text_you_must_set_the_view_your_complete_strava_profile_permission_when_you_authorize_the_app))
                                DefaultSpacer()
                                Text(text = stringResource(R.string.text_to_fix_this_disconnect_then_connect_again_and_leave_the_permission_enabled))
                            }
                        }
                    }
                }
            }

            // Intervals.icu
            AnimatedVisibility(
                visible = intervalsEnabled
            ) {
                Column {
                    DefaultSpacer()
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.intervals),
                        style = MaterialTheme.typography.bodyMedium
                    )
                    DefaultSpacer()
                    AnimatedVisibility(
                        visible = !isIntervalsConnected
                    ) {
                        val context = LocalContext.current
                        FullWidthButton(onClick = {
                            syncWithIntervalsViewModel.connect(context)
                        }) {
                            Text(text = stringResource(R.string.connect_with_intervals))
                        }
                    }
                    AnimatedVisibility(
                        visible = isIntervalsConnected
                    ) {
                        Column {
                            FullWidthButton(
                                enabled = isAllowedToReadAthleteSettings && bikesToShow.isNotEmpty(),
                                onClick = {
                                    syncWithIntervalsViewModel.updateDistancesFromIntervals(
                                        context = context,
                                        onResult = {
                                            // nothing to do
                                        },
                                        onError = { code ->
                                            // TODO
                                        }
                                    )
                                }
                            ) {
                                Text(text = stringResource(R.string.btn_text_update_distances_from_intervals))
                            }
                            DefaultSpacer()
                            FullWidthOutlinedButton(onClick = {
                                destinationsNavigator.navigate(SyncWithIntervalsDestination)
                            }) {
                                Text(text = stringResource(R.string.dropdown_item_sync_with_intervals))
                            }
                            DefaultSpacer()
                            IntervalsAttributionTextAndLogo()
                        }
                    }
                }
            }
        }
    }
}
