package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.bikes

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime

@HiltViewModel(assistedFactory = BikeEditViewModel.BikeEditViewModelFactory::class)
class BikeEditViewModel @AssistedInject constructor(
    @Assisted val bikeUid: Long,
    val repository: KJsRepository
) : ViewModel() {

    private val _bike: MutableLiveData<Bike?> = MutableLiveData()
    val bike: LiveData<Bike?> = _bike

    private val _numberOfComponentsForBike: MutableStateFlow<Int> = MutableStateFlow(0)
    val numberOfComponentsForBike: StateFlow<Int> = _numberOfComponentsForBike

    fun updateName(name: String) {
        if (bike.value != null) {
            _bike.value = bike.value!!.copy(
                name = name
            )
        }
    }

    fun updateIsElectric(isElectric: Boolean) {
        if (bike.value != null) {
            _bike.value = bike.value!!.copy(
                isElectric = isElectric
            )
        }
    }

    fun updateIsActive(isActive: Boolean) {
        if (bike.value != null) {
            _bike.value = bike.value!!.copy(
                isActive = isActive
            )
        }
    }

    fun updateMileage(mileage: Int) {
        if (bike.value != null) {
            _bike.value = bike.value!!.copy(
                mileage = mileage
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateBuildDate(millis: Long) {
        if (bike.value != null) {
            val buildDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
            val buildDate =
                buildDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
            _bike.value = bike.value!!.copy(
                buildDate = buildDate
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateLastUsedDate(millis: Long?) {
        if (bike.value != null) {
            if (millis != null) {
                val lastUsedDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val lastUsedDate =
                    lastUsedDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _bike.value = bike.value!!.copy(
                    lastUsedDate = lastUsedDate
                )
            } else {
                _bike.value = bike.value!!.copy(
                    lastUsedDate = null
                )
            }
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateRetirementDate(millis: Long?) {
        if (bike.value != null) {
            if (millis != null) {
                val retirementDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val retirementDate =
                    retirementDateInstant.toLocalDateTime(TimeZone.currentSystemDefault()).date
                _bike.value = bike.value!!.copy(
                    retirementDate = retirementDate
                )
            } else {
                _bike.value = bike.value!!.copy(
                    retirementDate = null
                )
            }
        }
    }

    fun updateStravaGearId(gearId: String?) {
        if (bike.value != null) {
            _bike.value = bike.value!!.copy(
                stravaGearId = gearId
            )
        }
    }

    fun updateIntervalsGearId(gearId: String?) {
        if (bike.value != null) {
            _bike.value == bike.value!!.copy(
                intervalsGearId = gearId
            )
        }
    }

    fun commitBike() {
        if (bike.value != null) {
            viewModelScope.launch {
                repository.updateBike(bike.value!!)
            }
        }
    }

    init {
        if (bikeUid > 0) {
            viewModelScope.launch {
                _bike.value = repository.getBikeByUid(bikeUid)
                if (bike.value != null) {
                    val num = repository.getNumberOfComponentsForBike(bike.value!!.uid)
                    _numberOfComponentsForBike.value = num
                }
            }
        }
    }

    @AssistedFactory
    interface BikeEditViewModelFactory {
        fun create(bikeUid: Long): BikeEditViewModel
    }
}