package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.bikes

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Done
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.LinkAnnotation
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.TextLinkStyles
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withLink
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorNullableWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultNumberFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.IntervalsAttributionTextAndLogo
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.BikeDeleteDestination
import com.ramcosta.composedestinations.generated.destinations.BikeEditDestination
import com.ramcosta.composedestinations.generated.destinations.ManageBikesDestination
import com.ramcosta.composedestinations.generated.destinations.ManageComponentsDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.LocalTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun BikeEdit(
    bikeUid: Long,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_edit_a_bike))

    val bikeEditViewModel =
        hiltViewModel<BikeEditViewModel, BikeEditViewModel.BikeEditViewModelFactory> { factory ->
            factory.create(bikeUid = bikeUid)
        }

    var modified by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        BikeEditDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            ),
            KJsAction(
                imageVector = Icons.Default.Delete,
                contentDescription = stringResource(R.string.btn_text_delete),
                onClick = {
                    destinationsNavigator.navigate(
                        BikeDeleteDestination(bikeUid = bikeUid)
                    )
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Done,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                bikeEditViewModel.commitBike()
                modified = false
                destinationsNavigator.popBackStack(
                    ManageBikesDestination, inclusive = false
                )
            },
            enabled = modified
        ),
        headline = stringResource(R.string.hdr_edit_a_bike),
        advancedFunctionality = advancedFunctionality
    ) {
        val bike by bikeEditViewModel.bike.observeAsState()
        val buildDateInstant = bike?.let {
            LocalDateTime(it.buildDate, LocalTime(12, 0, 0)).toInstant(
                TimeZone.currentSystemDefault()
            )
        }
        var selectedBuildDate = buildDateInstant?.toEpochMilliseconds()
        val lastUsedDateInstant = bike?.let {
            it.lastUsedDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(
                    TimeZone.currentSystemDefault()
                )
            }
        }
        var selectedLastUsedDate = lastUsedDateInstant?.toEpochMilliseconds()
        val retirementDateInstance = bike?.let {
            it.retirementDate?.let { it1 ->
                LocalDateTime(it1, LocalTime(12, 0, 0)).toInstant(TimeZone.currentSystemDefault())
            }
        }
        var retirementDate = retirementDateInstance?.toEpochMilliseconds()

        val numberOfComponentsForBike by bikeEditViewModel.numberOfComponentsForBike.collectAsStateWithLifecycle()

        val isStravaEnabled by kJsGlobalScaffoldViewModel.stravaEnabled.collectAsStateWithLifecycle()
        val isIntervalsEnabled by kJsGlobalScaffoldViewModel.intervalsEnabled.collectAsStateWithLifecycle()

        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            if (isStravaEnabled && bike?.stravaGearId?.isNotEmpty() == true) {
                Text(
                    modifier = Modifier
                        .padding(8.dp)
                        .align(Alignment.End),
                    text = buildAnnotatedString {
                        withLink(
                            LinkAnnotation.Url(
                                url = "https://www.strava.com/bikes/${bike?.stravaGearId}",
                                styles = TextLinkStyles(
                                    style = SpanStyle(color = MaterialTheme.colorScheme.primary)
                                )
                            )
                        ) {
                            append("View on Strava")
                        }
                    }
                )
            }

            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = "UID $bikeUid",
                    color = MaterialTheme.colorScheme.tertiary,
                    modifier = Modifier.padding(16.dp, 0.dp)
                )

                AnimatedVisibility(
                    visible = advancedFunctionality && numberOfComponentsForBike > 0
                ) {
                    TextButton(onClick = {
                        destinationsNavigator.navigate(ManageComponentsDestination(bike?.uid))
                    }) {
                        Text(text = stringResource(R.string.btn_text_manage_bike_components))
                    }
                }
            }
            DefaultSpacer()
            DefaultTextFieldWithSpacer(
                value = bike?.name ?: stringResource(R.string.placehldr_name),
                onValueChange = {
                    bikeEditViewModel.updateName(it)
                    modified = true
                },
                label = stringResource(R.string.lbl_bike_name),
            )
            TextAndSwitch(
                text = stringResource(R.string.lbl_is_an_ebike),
                checked = bike?.isElectric == true
            ) {
                bikeEditViewModel.updateIsElectric(it)
                modified = true
            }
            DefaultDateSelectorWithSpacer(
                selectedDate = selectedBuildDate,
                label = stringResource(R.string.lbl_build_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedBuildDate = it
                    if (it != null) {
                        bikeEditViewModel.updateBuildDate(it)
                        modified = true
                    }
                }
            )
            DefaultNumberFieldWithSpacer(
                value = (bike?.mileage ?: "").toString(),
                onValueChange = { value ->
                    bikeEditViewModel.updateMileage(value.toIntOrNull() ?: 0)
                    modified = true
                },
                label = stringResource(R.string.lbl_mileage),
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedLastUsedDate,
                label = stringResource(R.string.lbl_last_used_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedLastUsedDate = it
                    bikeEditViewModel.updateLastUsedDate(it)
                    modified = true
                }
            )
            TextAndSwitch(
                text = stringResource(R.string.is_active),
                checked = bike?.isActive == true
            ) {
                bikeEditViewModel.updateIsActive(it)
                modified = true
            }
            if (bike?.isActive != true) {
                DefaultDateSelectorNullableWithSpacer(
                    selectedDate = retirementDate,
                    label = stringResource(R.string.lbl_bike_retirement_date),
                    placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                    onDateSelected = {
                        retirementDate = it
                        bikeEditViewModel.updateRetirementDate(it)
                        modified = true
                    }
                )
            }
            DefaultSpacer()
            DefaultSpacer()
            if (isStravaEnabled) {
                DefaultTextFieldWithSpacer(
                    value = bike?.stravaGearId ?: stringResource(R.string.placehldr_strava_gear_id),
                    onValueChange = { value ->
                        bikeEditViewModel.updateStravaGearId(value)
                        modified = true
                    },
                    label = stringResource(R.string.lbl_strava_gear_id)
                )
                if (bike?.stravaGearId?.isNotEmpty() == true) {
                    DefaultSpacer()
                    Icon(
                        painter = painterResource(R.drawable.api_logo_pwrdby_strava_horiz_orange),
                        contentDescription = "Powered by Strava",
                        tint = Color.Unspecified,
                        modifier = Modifier
                            .height(24.dp)
                            .padding(8.dp, 0.dp)
                            .align(Alignment.End)
                    )
                }
            }
            if (isIntervalsEnabled) {
                DefaultTextFieldWithSpacer(
                    value = bike?.intervalsGearId
                        ?: stringResource(R.string.placehldr_strava_gear_id),
                    onValueChange = { value ->
                        bikeEditViewModel.updateIntervalsGearId(value)
                        modified = true
                    },
                    label = stringResource(R.string.lbl_intervals_gear_id)
                )
                if (bike?.intervalsGearId?.isNotEmpty() == true) {
                    DefaultSpacer()
                    IntervalsAttributionTextAndLogo()
                }
            }
        }
    }
}
