package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.bikes

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.Save
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultNumberFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.BikeAddDestination
import com.ramcosta.composedestinations.generated.destinations.ManageBikesDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun BikeAdd(
    bikeAddViewModel: BikeAddViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_add_bike))

    // input fields
    var name by remember { mutableStateOf("") }
    var selectedDate by remember { mutableStateOf<Long?>(null) }
    var mileage by remember { mutableIntStateOf(0) }
    var isElectric by remember { mutableStateOf(false) }

    var modified by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        BikeAddDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.Default.Clear,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Save,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                val bike = Bike(
                    name = name,
                    buildDate = Instant.fromEpochMilliseconds(selectedDate!!)
                        .toLocalDateTime(
                            TimeZone.currentSystemDefault()
                        ).date,
                    mileage = mileage,
                    lastUsedDate = null,
                    isElectric = isElectric,
                    isActive = true,
                    retirementDate = null,
                    stravaGearId = null,
                    intervalsGearId = null,
                    uid = 0 // autogenerate
                )
                bikeAddViewModel.saveNewBike(
                    bike = bike,
                )
                modified = false
                destinationsNavigator.navigateUp()
                destinationsNavigator.popBackStack(
                    ManageBikesDestination, inclusive = false
                )
            },
            enabled = modified && name.isNotBlank() && selectedDate != null
        ),
        headline = stringResource(R.string.hdr_add_bike),
        advancedFunctionality = advancedFunctionality
    ) {

        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            DefaultTextFieldWithSpacer(
                value = name,
                onValueChange = {
                    name = it
                    modified = true
                },
                label = stringResource(R.string.lbl_name),
            )
            TextAndSwitch(
                text = stringResource(R.string.lbl_is_an_ebike),
                checked = isElectric
            ) {
                isElectric = it
                modified = true
            }
            DefaultDateSelectorWithSpacer(
                selectedDate = selectedDate,
                label = stringResource(R.string.lbl_build_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedDate = it
                    modified = true
                }
            )
            DefaultNumberFieldWithSpacer(
                value = mileage.toString(),
                onValueChange = {
                    mileage = it.toIntOrNull() ?: 0
                    modified = true
                },
                label = stringResource(R.string.lbl_mileage),
            )
        }
    }
}
