package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.activities

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Done
import androidx.compose.material.icons.filled.Task
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.BikeSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.IconSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ActivityAddDestination
import com.ramcosta.composedestinations.generated.destinations.ActivityEditDestination
import com.ramcosta.composedestinations.generated.destinations.ManageActivitiesDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@OptIn(ExperimentalFoundationApi::class)
@Destination<RootGraph>
@Composable
fun ManageActivities(
    manageActivitiesViewModel: ManageActivitiesViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_manage_activities))

    val context = LocalContext.current

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        currentDestination = ManageActivitiesDestination,
        destinationsNavigator = destinationsNavigator,
        windowSizeClass = windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Add,
            contentDescription = stringResource(R.string.btn_text_add_activity),
            onClick = {
                destinationsNavigator.navigate(ActivityAddDestination)
            }
        ),
        headline = stringResource(R.string.hdr_manage_activities),
        advancedFunctionality = advancedFunctionality
    ) {
        val activities: List<Activity> by manageActivitiesViewModel.activities.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val showCompletedActivities by manageActivitiesViewModel.showCompletedActivities.collectAsStateWithLifecycle()

        val bikes: List<Bike> by manageActivitiesViewModel.observeBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val currentBike: Bike? by manageActivitiesViewModel.currentBike.collectAsStateWithLifecycle(
            initialValue = null
        )

        val filteredActivities = activities.filter { activity ->
            (showCompletedActivities || activity.isCompleted == false) && (currentBike == null || currentBike!!.uid == activity.bikeUid)
        }

        Column(
            modifier = Modifier.fillMaxSize()
        ) {
            TextAndSwitch(
                text = stringResource(R.string.show_completed_activities),
                checked = showCompletedActivities
            ) {
                manageActivitiesViewModel.updateShowCompletedActivities(it)
            }
            BikeSelector(
                modifier = Modifier.padding(8.dp, 0.dp),
                currentBike = currentBike,
                label = stringResource(R.string.lbl_filter_by_bike),
                onBikeSelected = {
                    manageActivitiesViewModel.updateCurrentBike(it)
                },
                bikes = bikes
            )
            LazyColumn(
                modifier = Modifier
                    .fillMaxWidth()
            ) {
                stickyHeader {
                    Text(
                        modifier = Modifier
                            .fillMaxWidth()
                            .background(MaterialTheme.colorScheme.primaryContainer)
                            .padding(8.dp),
                        text = stringResource(R.string.hdr_activities),
                        style = MaterialTheme.typography.bodyMedium,
                    )
                }

                items(filteredActivities, key = { "activity.${it.uid}" }) { activity ->
                    Surface(
                        modifier = Modifier
                            .combinedClickable(
                                onClick = {
                                    destinationsNavigator.navigate(ActivityEditDestination(activity.uid))
                                },
                            )
                            .padding(4.dp),
                    ) {
                        Row(
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            AnimatedVisibility(
                                visible = activity.isCompleted
                            ) {
                                Icon(
                                    imageVector = Icons.Default.Done,
                                    contentDescription = stringResource(R.string.activity),
                                )
                            }
                            AnimatedVisibility(
                                visible = !activity.isCompleted
                            ) {
                                Icon(
                                    imageVector = Icons.Default.Task,
                                    contentDescription = stringResource(R.string.activity),
                                )
                            }
                            IconSpacer()
                            Text(
                                text = activity.title,
                                modifier = Modifier.weight(1f)
                            )
                            DefaultSpacer()
                            Text(
                                text = if (activity.dueDate != null) {
                                    activity.dueDate.toString()
                                } else {
                                    RideLevel.getLabel(context, activity.rideLevel)
                                }
                            )
                        }
                    }
                }
            }
        }
    }
}