package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.activities

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime

@HiltViewModel(assistedFactory = ActivityEditViewModel.ActivityEditViewModelFactory::class)
class ActivityEditViewModel @AssistedInject constructor(
    @Assisted val activityUid: Long,
    userPreferencesManager: UserPreferencesManager,
    val repository: KJsRepository
) : ViewModel() {

    private val _activity: MutableLiveData<Activity> = MutableLiveData()
    val activity: LiveData<Activity> = _activity

    val observeBikes = repository.observeBikes
    private val _currentBike: MutableStateFlow<Bike?> = MutableStateFlow(null)
    val currentBike: StateFlow<Bike?> = _currentBike

    val advancedFunctionality: StateFlow<Boolean> =
        userPreferencesManager.advancedFunctionality().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )
    val observeComponents = repository.observeComponents
    private val _currentComponent: MutableStateFlow<Component?> = MutableStateFlow(null)
    val currentComponent: StateFlow<Component?> = _currentComponent

    init {
        if (activityUid > 0) {
            viewModelScope.launch {
                _activity.value = repository.getActivityByUid(activityUid)
                val tempActivity = activity.value
                if (tempActivity != null) {
                    if (tempActivity.bikeUid != null && tempActivity.bikeUid > 0) {
                        _currentBike.value = repository.getBikeByUid(tempActivity.bikeUid)
                    }
                    if (tempActivity.componentUid != null && tempActivity.componentUid > 0) {
                        _currentComponent.value = repository.getComponentByUid(tempActivity.componentUid)
                    }
                }
            }
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateTitle(title: String) {
        if (activity.value != null) {
            _activity.value = activity.value!!.copy(
                title = title
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateDescription(description: String) {
        if (activity.value != null) {
            _activity.value = activity.value!!.copy(
                description = description
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateIsCompleted(isCompleted: Boolean) {
        if (activity.value != null) {
            _activity.value = activity.value!!.copy(
                isCompleted = isCompleted
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateIsEBikeSpecific(isEBikeSpecific: Boolean) {
        if (activity.value != null) {
            _activity.value = activity.value!!.copy(
                isEBikeSpecific = isEBikeSpecific
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateCreatedDate(millis: Long) {
        if (activity.value != null) {
            val buildDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
            _activity.value = activity.value!!.copy(
                createdInstant = buildDateInstant
            )
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateDueDate(millis: Long?) {
        if (activity.value != null) {
            if (millis != null) {
                val buildDateInstant = Instant.Companion.fromEpochMilliseconds(millis)
                val buildDate =
                    buildDateInstant.toLocalDateTime(TimeZone.Companion.currentSystemDefault()).date
                _activity.value = activity.value!!.copy(
                    dueDate = buildDate
                )
            } else {
                _activity.value = activity.value!!.copy(
                    dueDate = null
                )
            }
        }
    }

    fun updateAttachedBike(bikeUid: Long?) {
        if (bikeUid != null) {
            viewModelScope.launch {
                _currentBike.value = repository.getBikeByUid(bikeUid)
            }
        } else {
            _currentBike.value = null
        }
    }

    fun updateAttachedComponent(componentUid: Long?) {
        if (componentUid != null) {
            viewModelScope.launch {
                _currentComponent.value = repository.getComponentByUid(componentUid)
            }
        } else {
            _currentComponent.value = null
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateRideLevel(level: Int?) {
        viewModelScope.launch {
            _activity.value = activity.value!!.copy(
                rideLevel = level
            )
        }
    }

    fun commitActivity() {
        if (activity.value != null) {
            viewModelScope.launch {
                repository.updateActivity(activity.value!!)
            }
        }
    }

    @AssistedFactory
    interface ActivityEditViewModelFactory {
        fun create(activityUid: Long): ActivityEditViewModel
    }
}