package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.activities

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Done
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultBikeSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultComponentSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorNullableWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultInstantSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultRideLevelSelectorActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ActivityDeleteDestination
import com.ramcosta.composedestinations.generated.destinations.ActivityEditDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.LocalTime
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toInstant
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun ActivityEdit(
    activityUid: Long,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_edit_an_activity))

    val activityEditViewModel =
        hiltViewModel<ActivityEditViewModel, ActivityEditViewModel.ActivityEditViewModelFactory> { factory ->
            factory.create(activityUid = activityUid)
        }

    var modified by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ActivityEditDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            ),
            KJsAction(
                imageVector = Icons.Default.Delete,
                contentDescription = stringResource(R.string.btn_text_delete),
                onClick = {
                    destinationsNavigator.navigate(ActivityDeleteDestination(activityUid = activityUid))
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Done,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                activityEditViewModel.commitActivity()
                modified = false
                destinationsNavigator.navigateUp()
            },
            enabled = modified
        ),
        headline = stringResource(R.string.hdr_edit_an_activity),
        advancedFunctionality = advancedFunctionality
    ) {

        val activity by activityEditViewModel.activity.observeAsState()
        var selectedCreatedDate = activity?.createdInstant?.toEpochMilliseconds()
        val dueDateInstant = activity?.dueDate?.let {
            LocalDateTime(
                it,
                LocalTime(12, 0, 0)
            ).toInstant(TimeZone.currentSystemDefault())
        }
        var selectedDueDate = dueDateInstant?.toEpochMilliseconds()

        val bikes: List<Bike> by activityEditViewModel.observeBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val currentBike: Bike? by activityEditViewModel.currentBike.collectAsStateWithLifecycle(
            initialValue = null
        )

        val advancedFunctionality by activityEditViewModel.advancedFunctionality.collectAsStateWithLifecycle()

        val components: List<Component> by activityEditViewModel.observeComponents.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val currentComponent: Component? by activityEditViewModel.currentComponent.collectAsStateWithLifecycle(
            initialValue = null
        )

        Column(
            modifier = Modifier.verticalScroll(rememberScrollState())
        ) {
            DefaultTextFieldWithSpacer(
                value = activity?.title ?: "",
                label = stringResource(R.string.lbl_activity_title),
                placeholder = stringResource(R.string.placehldr_title),
                onValueChange = {
                    activityEditViewModel.updateTitle(it)
                    modified = true
                },
            )
            DefaultTextFieldWithSpacer(
                value = activity?.description ?: "",
                onValueChange = {
                    activityEditViewModel.updateDescription(it)
                    modified = true
                },
                label = stringResource(R.string.lbl_description),
            )
            DefaultRideLevelSelectorActivity(
                rideLevel = activity?.rideLevel,
                onItemSelected = {
                    activityEditViewModel.updateRideLevel(it)
                    modified = true
                }
            )
            TextAndSwitch(
                text = stringResource(R.string.is_ebike_specific),
                checked = activity?.isEBikeSpecific == true
            ) {
                activityEditViewModel.updateIsEBikeSpecific(it)
                modified = true
            }
            TextAndSwitch(
                text = stringResource(R.string.lbl_completed),
                checked = activity?.isCompleted == true
            ) {
                activityEditViewModel.updateIsCompleted(it)
                modified = true
            }
            DefaultBikeSelectorWithSpacer(
                value = if (currentBike != null) {
                    currentBike!!.name
                } else {
                    stringResource(R.string.dropdown_item_none)
                },
                label = stringResource(R.string.lbl_attached_to_bike),
                onMenuItemClick = {
                    activityEditViewModel.updateAttachedBike(it)
                    modified = true
                },
                bikes = bikes
            )
            AnimatedVisibility(
                visible = advancedFunctionality
            ) {
                DefaultComponentSelectorWithSpacer(
                    value = if (currentComponent != null) {
                        currentComponent!!.name
                    } else {
                        stringResource(R.string.dropdown_item_none)
                    },
                    label = stringResource(R.string.lbl_for_component),
                    onMenuItemClick = {
                        // activityEditViewModel
                        modified = true
                    },
                    components = components
                )
            }
            DefaultInstantSelectorWithSpacer(
                selectedDate = selectedCreatedDate,
                label = stringResource(R.string.lbl_created_at),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd_hh_mm_ss),
                onDateSelected = {
                    selectedCreatedDate = it
                    if (it != null) {
                        activityEditViewModel.updateCreatedDate(it)
                        modified = true
                    }
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedDueDate,
                label = stringResource(R.string.lbl_due_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedDueDate = it
                    activityEditViewModel.updateDueDate(it)
                    modified = true
                }
            )
        }
    }
}
