package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.activities

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.Save
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultBikeSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultComponentSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorNullableWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultDateSelectorWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultRideLevelSelectorActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultTextFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ActivityAddDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlin.time.Clock
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun ActivityAdd(
    activityAddViewModel: ActivityAddViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_add_activity))

    var title by remember { mutableStateOf("") }
    var description by remember { mutableStateOf("") }
    var isCompleted by remember { mutableStateOf(false) }
    var selectedCreatedDate by remember {
        mutableLongStateOf(
            Clock.System.now().toEpochMilliseconds()
        )
    }
    var selectedDueDate by remember { mutableStateOf<Long?>(null) }
    var isEBikeSpecific by remember { mutableStateOf(false) }
    var rideLevel: Int? by remember { mutableStateOf(null) }
    val bikes: List<Bike> by activityAddViewModel.observeBikes.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val currentBike: Bike? by activityAddViewModel.currentBike.collectAsStateWithLifecycle(
        initialValue = null
    )

    val components: List<Component> by activityAddViewModel.observeComponents.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )
    val currentComponent: Component? by activityAddViewModel.currentComponent.collectAsStateWithLifecycle(
        initialValue = null
    )

    var modified by remember { mutableStateOf(false) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        ActivityAddDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.Default.Clear,
                contentDescription = stringResource(R.string.btn_text_cancel),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            imageVector = Icons.Default.Save,
            contentDescription = stringResource(R.string.btn_text_save),
            onClick = {
                val activity = Activity(
                    title = title,
                    description = description,
                    rideUid = null,
                    isEBikeSpecific = isEBikeSpecific,
                    rideLevel = null, // TODO
                    createdInstant = Instant.fromEpochMilliseconds(
                        selectedCreatedDate
                    ),
                    isCompleted = isCompleted,
                    dueDate = if (selectedDueDate != null) {
                        Instant.fromEpochMilliseconds(selectedDueDate!!)
                            .toLocalDateTime(
                                TimeZone.currentSystemDefault()
                            ).date
                    } else {
                        null
                    },
                    doneInstant = null,
                    componentUid = null,
                    bikeUid = currentBike?.uid
                )
                activityAddViewModel.saveActivity(activity)
                modified = false
                destinationsNavigator.navigateUp()
            },
            enabled = modified && title.isNotBlank()
        ),
        headline = stringResource(R.string.hdr_add_activity),
        advancedFunctionality = advancedFunctionality
    ) {
        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            DefaultTextFieldWithSpacer(
                value = title,
                onValueChange = {
                    title = it
                    modified = true
                },
                label = stringResource(R.string.lbl_activity_title),
            )
            DefaultTextFieldWithSpacer(
                value = description,
                onValueChange = {
                    description = it
                    modified = true
                },
                label = stringResource(R.string.lbl_description),
            )
            DefaultRideLevelSelectorActivity(
                rideLevel = rideLevel
            ) {
                rideLevel = it
                modified = true
            }
            TextAndSwitch(
                text = stringResource(R.string.is_ebike_specific),
                checked = isEBikeSpecific
            ) {
                isEBikeSpecific = it
                modified = true
            }
            TextAndSwitch(
                text = stringResource(R.string.lbl_completed),
                checked = isCompleted
            ) {
                isCompleted = it
                modified = true
            }
            DefaultBikeSelectorWithSpacer(
                value = if (currentBike != null) {
                    currentBike!!.name
                } else {
                    stringResource(R.string.dropdown_item_none)
                },
                label = stringResource(R.string.lbl_attached_to_bike),
                onMenuItemClick = {
                    activityAddViewModel.updateAttachedBike(it)
                    modified = true
                },
                bikes = bikes
            )
            AnimatedVisibility(
                visible = advancedFunctionality
            ) {
                DefaultComponentSelectorWithSpacer(
                    value = if (currentComponent != null) {
                        currentComponent!!.name
                    } else {
                        stringResource(R.string.dropdown_item_none)
                    },
                    label = stringResource(R.string.lbl_for_component),
                    onMenuItemClick = {
                        activityAddViewModel.updateAttachedComponent(it)
                        modified = true
                    },
                    components = components
                )
            }
            DefaultDateSelectorWithSpacer(
                selectedDate = selectedCreatedDate,
                label = stringResource(R.string.lbl_created_at),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    if (it != null) {
                        selectedCreatedDate = it
                        modified = true
                    }
                }
            )
            DefaultDateSelectorNullableWithSpacer(
                selectedDate = selectedDueDate,
                label = stringResource(R.string.lbl_due_date),
                placeholder = stringResource(R.string.placehldr_yyyy_mm_dd),
                onDateSelected = {
                    selectedDueDate = it
                    modified = true
                }
            )
        }
    }
}