package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike

import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.loadTemplateActivitiesJsonAndImport
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.Theme
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.SupportedLanguages
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import javax.inject.Inject

enum class DistanceMeasure(val metric: String) {
    KM("km"),
    MI("mi")
}

@HiltViewModel
class SettingsViewModel @Inject constructor(
    private val userPreferencesManager: UserPreferencesManager,
    val repository: KJsRepository,
) : ViewModel() {

    private val _doesThirdPartyDataExist: MutableStateFlow<Boolean> = MutableStateFlow(false)
    val doesThirdPartyDataExist: StateFlow<Boolean> = _doesThirdPartyDataExist

    init {
        viewModelScope.launch {
            _doesThirdPartyDataExist.value =
                repository.getNumberOfBikesWithThirdPartyData() > 0 || repository.getNumberOfImportedGearStrava() > 0 || repository.getNumberOfImportedGearIntervals() > 0
        }
    }

    val userSelectedTheme: StateFlow<Theme> = userPreferencesManager.theme().stateIn(
        viewModelScope,
        SharingStarted.Companion.WhileSubscribed(),
        Theme.Auto
    )

    fun updateUserSelectedTheme(newTheme: Theme) {
        viewModelScope.launch {
            userPreferencesManager.setTheme(newTheme)
        }
    }

    val distancesMetric: StateFlow<DistanceMeasure> =
        userPreferencesManager.distanceMeasure().stateIn(
            viewModelScope,
            SharingStarted.Companion.WhileSubscribed(),
            DistanceMeasure.KM
        )

    fun updateDistancesUseKm(newUseKm: DistanceMeasure) {
        viewModelScope.launch {
            userPreferencesManager.setDistanceMeasure(newUseKm)
        }
    }

    val stravaEnabled: StateFlow<Boolean> = userPreferencesManager.stravaEnabled().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    fun updateStravaEnabled(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setStravaEnabled(enabled)
        }
    }

    val advancedFunctionality: StateFlow<Boolean> =
        userPreferencesManager.advancedFunctionality().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )

    fun updateAdvancedFunctionality(advanced: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setAdvancedFunctionality(advanced)
        }
    }

    val intervalsEnabled: StateFlow<Boolean> = userPreferencesManager.intervalsEnabled().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    fun updateIntervalsEnabled(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setIntervalsEnabled(enabled)
        }
    }

    val templateActivityLanguage: StateFlow<String> =
        userPreferencesManager.templateActivityLanguage().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            "default" // meaning use the system language
        )

    fun updateTemplateActivityLanguage(language: String) {
        viewModelScope.launch {
            userPreferencesManager.setTemplateActivityLanguage(language)
        }
    }

    val automaticallyUpdateTemplateActivitiesOnAppStart: StateFlow<Boolean> =
        userPreferencesManager.automaticallyUpdateTemplatesOnAppStart().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            true
        )

    fun updateAutomaticallyUpdateTemplateActivitiesOnAppStart(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setAutomaticallyUpdateTemplateActivitiesOnAppStart(enabled)
        }
    }

    val updateBikeDistancesPeriodically: StateFlow<Boolean> =
        userPreferencesManager.updateBikeDistancesPeriodically().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )

    fun updateUpdateBikeDistancesPeriodically(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setUpdateBikeDistancesPeriodically(enabled)
        }
    }

    val notificationsEnabled: StateFlow<Boolean> =
        userPreferencesManager.enableNotifications().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )

    fun updateNotificationsEnabled(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setEnableNotifications(enabled)
        }
    }

    val automaticallyExportData: StateFlow<Boolean> =
        userPreferencesManager.automaticallyExportData().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )
    fun updateAutomaticallyExportData(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setAutomaticallyExportData(enabled)
        }
    }

    val automaticallyPruneExportedData: StateFlow<Boolean> =
        userPreferencesManager.automaticallyPruneExportedData().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            true
        )
    fun updateAutomaticallyPruneExportedData(enabled: Boolean) {
        viewModelScope.launch {
            userPreferencesManager.setAutomaticallyPruneExportedData(enabled)
        }
    }

    val automaticExportPruningLimit: StateFlow<Int> =
        userPreferencesManager.automaticExportPruningLimit().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            10
        )
    fun updateAutomaticExportPruningLimit(limit: Int) {
        viewModelScope.launch {
            userPreferencesManager.setAutomaticExportPruningLimit(limit)
        }
    }

    fun replaceTemplateActivitiesWithNewLanguage(language: String) {
        var languageChecked = language
        // check whether that language is OK
        if (!SupportedLanguages.getListOfSupportedLanguages().contains(language)) {
            // it is not known, so let's go to system language, or en
            languageChecked = SupportedLanguages.determineLocaleToLoad()
        }
        loadTemplateActivitiesJsonAndImport(
            language = languageChecked,
            importDataCallback = { newTemplateActivities ->
                viewModelScope.launch {
                    // clear out templates
                    repository.deleteAllBuiltInTemplateActivities()
                    // import List
                    newTemplateActivities.forEach { templateActivity ->
                        repository.insertTemplateActivity(templateActivity)
                    }
                }
            },
            errorCallback = {
                Log.d("ReplaceTemplatesWithNewLanguage", "Unable to import templates")
            }
        )
    }

    fun cleanThirdPartyData() {
        viewModelScope.launch {
            val allBikes = repository.getAllBikes()
            Log.d("Clean 3rd-party data", "Will clean ${allBikes.size} bikes...")
            allBikes.forEach { bike ->
                var bikeHasChanged = false
                var newBike = bike.copy()
                if (bike.intervalsGearId?.isNotEmpty() == true) {
                    newBike = newBike.copy(
                        intervalsGearId = null
                    )
                    bikeHasChanged = true
                }
                if (bike.stravaGearId?.isNotEmpty() == true) {
                    newBike = newBike.copy(
                        stravaGearId = null
                    )
                    bikeHasChanged = true
                }
                Log.d("Clean 3rd-party data", "Bike cleaned $newBike")
                if (bikeHasChanged) {
                    repository.updateBike(newBike)
                }
            }
            Log.d("Clean 3rd-party data", "Will clean Summary Gear tables...")
            repository.deleteAllIntervalsSummaryGear()
            repository.deleteAllStravaSummaryGear()
        }
    }
}