package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike

import android.Manifest
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Dataset
import androidx.compose.material.icons.filled.DeleteForever
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.Theme
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultNumberFieldWithSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthOutlinedButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.IconSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.LanguageSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndSwitch
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.TextAndTwoStateToggle
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.ThemeTriStateToggle
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithIntervalsViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithStravaViewModel
import com.google.accompanist.permissions.ExperimentalPermissionsApi
import com.google.accompanist.permissions.PermissionStatus
import com.google.accompanist.permissions.rememberPermissionState
import com.google.accompanist.permissions.shouldShowRationale
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ManageTemplateActivitiesDestination
import com.ramcosta.composedestinations.generated.destinations.SettingsDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@OptIn(ExperimentalPermissionsApi::class)
@Destination<RootGraph>
@Composable
fun Settings(
    settingsViewModel: SettingsViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    syncWithStravaViewModel: SyncWithStravaViewModel = hiltViewModel(),
    syncWithIntervalsViewModel: SyncWithIntervalsViewModel = hiltViewModel(),
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_settings))

    val notificationPermissionState =
        rememberPermissionState(Manifest.permission.POST_NOTIFICATIONS)

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        SettingsDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.navigateUp()
                }
            )
        ),
        headline = stringResource(R.string.hdr_settings),
        advancedFunctionality = advancedFunctionality,
        helpUrlString = "https://jan-exner.de/software/android/jkbikemdp/manual/settings"
    ) {
        val userSelectedTheme by settingsViewModel.userSelectedTheme.collectAsStateWithLifecycle()
        val distancesMeasure by settingsViewModel.distancesMetric.collectAsStateWithLifecycle()

        val stravaEnabled by settingsViewModel.stravaEnabled.collectAsStateWithLifecycle()
        val isStravaConnected by syncWithStravaViewModel.isStravaConnected.collectAsStateWithLifecycle()

        val intervalsEnabled by settingsViewModel.intervalsEnabled.collectAsStateWithLifecycle()
        val isIntervalsConnected by syncWithIntervalsViewModel.isIntervalsConnected.collectAsStateWithLifecycle()

        val advancedFunctionality by settingsViewModel.advancedFunctionality.collectAsStateWithLifecycle()

        val automaticallyUpdateTemplateActivitiesOnAppStart by settingsViewModel.automaticallyUpdateTemplateActivitiesOnAppStart.collectAsStateWithLifecycle()
        val templateActivityLanguage by settingsViewModel.templateActivityLanguage.collectAsStateWithLifecycle()

        val updateBikeDistancesPeriodically by settingsViewModel.updateBikeDistancesPeriodically.collectAsStateWithLifecycle()

        val enableNotifications by settingsViewModel.notificationsEnabled.collectAsStateWithLifecycle()

        val automaticallyExportData by settingsViewModel.automaticallyExportData.collectAsStateWithLifecycle()
        val automaticallyPruneExportedData by settingsViewModel.automaticallyPruneExportedData.collectAsStateWithLifecycle()
        val automaticExportPruningLimit by settingsViewModel.automaticExportPruningLimit.collectAsStateWithLifecycle()

        val doesThirdPartyDataExist by settingsViewModel.doesThirdPartyDataExist.collectAsStateWithLifecycle()

        val context = LocalContext.current

        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            Text(
                text = stringResource(R.string.title_section_ui_visuals),
                style = MaterialTheme.typography.titleMedium,
                modifier = Modifier.padding(8.dp)
            )
            ThemeTriStateToggle(
                text = stringResource(R.string.lbl_theme),
                currentTheme = userSelectedTheme,
                updateTheme = { it: Theme ->
                    settingsViewModel.updateUserSelectedTheme(it)
                }
            )
            DefaultSpacer()
            HorizontalDivider()
            DefaultSpacer()
            Text(
                text = stringResource(R.string.title_section_advanced_functionality),
                style = MaterialTheme.typography.titleMedium,
                modifier = Modifier.padding(8.dp)
            )
            DefaultSpacer()
            TextAndSwitch(
                text = stringResource(R.string.show_advanced_functionality),
                checked = advancedFunctionality
            ) {
                settingsViewModel.updateAdvancedFunctionality(it)
            }
            DefaultSpacer()
            TextAndSwitch(
                text = stringResource(R.string.switch_automatically_export_data_as_backup),
                checked = automaticallyExportData
            ) {
                settingsViewModel.updateAutomaticallyExportData(it)
            }
            AnimatedVisibility(
                visible = automaticallyExportData
            ) {
                Column {
                    DefaultSpacer()
                    TextAndSwitch(
                        text = stringResource(R.string.switch_prune_automatic_exports),
                        checked = automaticallyPruneExportedData
                    ) {
                        settingsViewModel.updateAutomaticallyPruneExportedData(it)
                    }
                }
            }
            AnimatedVisibility(
                visible = automaticallyExportData && automaticallyPruneExportedData
            ) {
                Column {
                    DefaultSpacer()
                    DefaultNumberFieldWithSpacer(
                        modifier = Modifier.padding(16.dp, 0.dp),
                        value = automaticExportPruningLimit.toString(),
                        label = stringResource(R.string.switch_keep_how_many_exports),
                        enabled = true,
                        placeholder = stringResource(R.string.placeholder_10_is_a_good_value)
                    ) {
                        settingsViewModel.updateAutomaticExportPruningLimit(it.toIntOrNull() ?: 10)
                    }
                }
            }
            DefaultSpacer()
            HorizontalDivider()
            DefaultSpacer()
            Text(
                text = stringResource(R.string.title_section_language),
                style = MaterialTheme.typography.titleMedium,
                modifier = Modifier.padding(8.dp)
            )
            DefaultSpacer()
            TextAndTwoStateToggle(
                text = stringResource(R.string.lbl_distances_use_metric),
                currentTheme = distancesMeasure,
                updateTheme = { it: DistanceMeasure ->
                    settingsViewModel.updateDistancesUseKm(it)
                }
            )
            DefaultSpacer()
            TextAndSwitch(
                text = stringResource(R.string.automatically_update_template_activities_on_app_start),
                checked = automaticallyUpdateTemplateActivitiesOnAppStart
            ) {
                settingsViewModel.updateAutomaticallyUpdateTemplateActivitiesOnAppStart(it)
            }
            DefaultSpacer()
            LanguageSelector(
                currentLanguage = templateActivityLanguage,
                labelText = stringResource(R.string.lbl_get_template_activities_in_language)
            ) { languageCode ->
                if (languageCode.isNullOrEmpty()) {
                    // don't change anything
                } else {
                    settingsViewModel.updateTemplateActivityLanguage(languageCode)
                    settingsViewModel.replaceTemplateActivitiesWithNewLanguage(languageCode)
                }
            }
            DefaultSpacer()
            Text(
                text = stringResource(R.string.you_can_add_your_own_template_activities),
                modifier = Modifier.padding(16.dp, 0.dp)
            )
            DefaultSpacer()
            OutlinedButton(
                modifier = Modifier
                    .padding(8.dp, 0.dp)
                    .align(Alignment.End),
                onClick = {
                    destinationsNavigator.navigate(ManageTemplateActivitiesDestination)
                }
            ) {
                Icon(
                    imageVector = Icons.Default.Dataset,
                    contentDescription = stringResource(R.string.btn_text_manage_template_activities)
                )
                IconSpacer()
                Text(text = stringResource(R.string.btn_text_manage_template_activities))
            }
            DefaultSpacer()
            HorizontalDivider()
            DefaultSpacer()
            Text(
                text = stringResource(R.string.title_section_integrations),
                style = MaterialTheme.typography.titleMedium,
                modifier = Modifier.padding(8.dp)
            )
            // STRAVA
//            DefaultSpacer()
//            TextAndSwitch(
//                text = stringResource(R.string.enable_strava_integration),
//                checked = stravaEnabled
//            ) {
//                settingsViewModel.updateStravaEnabled(it)
//                if (!it) { // disabled, so let's disconnect
//                    syncWithStravaViewModel.disconnectStrava()
//                } else { // enabled, so we may have to switch off the other(s)
//                    if (intervalsEnabled) {
//                        settingsViewModel.updateIntervalsEnabled(false)
//                        syncWithIntervalsViewModel.disconnect()
//                    }
//                }
//            }
            DefaultSpacer()
            // Intervals.icu
            TextAndSwitch(
                text = stringResource(R.string.enable_intervals_integration),
                checked = intervalsEnabled
            ) {
                settingsViewModel.updateIntervalsEnabled(it)
                if (!it) { // disabled, so let's disconnect
                    syncWithIntervalsViewModel.disconnect()
                } else { // enabled, so we may have to switch off the other(s)
                    if (stravaEnabled) {
                        settingsViewModel.updateStravaEnabled(false)
                        syncWithStravaViewModel.disconnectStrava()
                    }
                }
            }
            AnimatedVisibility(
                visible = stravaEnabled && !isStravaConnected
            ) {
                Column(
                    modifier = Modifier.fillMaxWidth()
                ) {
                    DefaultSpacer()
                    Icon(
                        painter = painterResource(R.drawable.btn_strava_connect_with_orange),
                        contentDescription = stringResource(R.string.connect_with_strava),
                        tint = Color.Unspecified,
                        modifier = Modifier
                            .align(Alignment.CenterHorizontally)
                            .clickable(
                                enabled = stravaEnabled,
                                onClick = {
                                    syncWithStravaViewModel.connect(context)
                                }
                            )
                    )
                }
            }
            AnimatedVisibility(
                visible = intervalsEnabled && !isIntervalsConnected
            ) {
                Column {
                    DefaultSpacer()
                    FullWidthButton(
                        enabled = intervalsEnabled,
                        onClick = {
                            syncWithIntervalsViewModel.connect(context)
                        }
                    ) {
                        Text(text = stringResource(R.string.connect_with_intervals))
                    }
                }
            }
            AnimatedVisibility(
                visible = stravaEnabled || intervalsEnabled
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.periodically_update_bike_data),
                    checked = updateBikeDistancesPeriodically
                ) {
                    settingsViewModel.updateUpdateBikeDistancesPeriodically(it)
                }
            }
            AnimatedVisibility(
                visible = advancedFunctionality && updateBikeDistancesPeriodically
            ) {
                TextAndSwitch(
                    text = stringResource(R.string.enable_notifications),
                    checked = enableNotifications
                ) {
                    settingsViewModel.updateNotificationsEnabled(it)
                }
            }
            AnimatedVisibility(
                visible = advancedFunctionality && updateBikeDistancesPeriodically && enableNotifications
            ) {
                Column(
                    modifier = Modifier.padding(16.dp)
                ) {
                    if (notificationPermissionState.status == PermissionStatus.Granted) {
                        Text(text = stringResource(R.string.you_have_granted_permission_for_this_app_to_show_notifications))
                    } else {
                        // show a button
                        if (notificationPermissionState.status.shouldShowRationale) {
                            Text(text = stringResource(R.string.text_rationale_for_enable_notifications))
                        } else {
                            Text(text = stringResource(R.string.text_please_enable_notifications_for_this_app))
                        }
                        DefaultSpacer()
                        FullWidthOutlinedButton(onClick = {
                            notificationPermissionState.launchPermissionRequest()
                        }) {
                            Text(text = stringResource(R.string.btn_text_enable_notifications))
                        }
                    }
                    DefaultSpacer()
                }
            }
            AnimatedVisibility(
                visible = !stravaEnabled && !intervalsEnabled && doesThirdPartyDataExist
            ) {
                Column(
                    modifier = Modifier.padding(16.dp, 0.dp)
                ) {
                    DefaultSpacer()
                    Text(text = stringResource(R.string.if_you_had_previously_enabled_any_integration_and_used_it_the_local_bike_database_may_contain_gear_ids_for_some_of_your_bikes))
                    DefaultSpacer()
                    Text(text = stringResource(R.string.you_can_use_the_button_below_to_clean_up_that_data_no_bikes_will_be_deleted))
                    OutlinedButton(
                        modifier = Modifier
                            .align(Alignment.End),
                        onClick = {
                            settingsViewModel.cleanThirdPartyData()
                        }
                    ) {
                        Icon(
                            imageVector = Icons.Default.DeleteForever,
                            contentDescription = stringResource(R.string.btn_text_delete)
                        )
                        IconSpacer()
                        Text(text = stringResource(R.string.btn_text_delete))
                    }
                }
            }
        }
    }
}