package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Component
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.WearLevel
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.todayIn
import javax.inject.Inject
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@HiltViewModel
class HomeViewModel @Inject constructor(
    private val userPreferencesManager: UserPreferencesManager,
    private val repository: KJsRepository
) : ViewModel() {

    val observeActivityWithBikeData = repository.observeActivityWithBikeAndRideDataOrderedByDueDate

    val observeBikes = repository.observeBikes

    val observeRides = repository.observeRides

    val observeRidesInPrepararion = repository.observeRidesInPreparation

    val advancedFunctionality: StateFlow<Boolean> =
        userPreferencesManager.advancedFunctionality().stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(),
            false
        )

    private val componentsLookupTable: MutableMap<Long, Component> = mutableMapOf()

    init {
        viewModelScope.launch {
            val components = repository.getAllComponents()
            components.forEach { component ->
                componentsLookupTable.put(component.uid, component)
            }
        }
    }

    fun getComponentNameForUid(componentUid: Long?): String? {
        if (componentUid == null) {
            return null
        }
        return componentsLookupTable[componentUid]?.name
    }

    fun updateActivity(activity: Activity) {
        viewModelScope.launch {
            repository.updateActivity(activity)
        }
    }

    @OptIn(ExperimentalTime::class)
    fun updateAttachedComponent(
        componentUid: Long,
        bikeUid: Long?
    ) {
        viewModelScope.launch {
            val bikeMileage = if (bikeUid != null) {
                repository.getBikeByUid(bikeUid)?.mileage ?: 0
            } else {
                0
            }
            val component = repository.getComponentByUid(componentUid)?.copy(
                lastCheckDate = Clock.System.todayIn(TimeZone.Companion.currentSystemDefault()),
            )
            val newComponent = if (bikeMileage > 0) {
                component?.copy(
                    lastCheckMileage = bikeMileage - (component.mileageOffset ?: 0)
                )
            } else {
                component
            }
            if (newComponent != null) {
                repository.updateComponent(newComponent)
            }
        }
    }

    fun logComponentWearLevel(componentUid: Long, wearLevel: WearLevel) {
        viewModelScope.launch {
            val component = repository.getComponentByUid(componentUid)
            if (component != null) {
                val updatedComponent = component.copy(
                    wearLevel = wearLevel
                )
                repository.updateComponent(updatedComponent)
            }
        }
    }
}
