package com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Done
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowHeightSizeClass
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.material3.windowsizeclass.WindowWidthSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.LineBreak
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Ride
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ActivityWithBikeAndRideData
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.ComponentWearLevelDialog
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.FullWidthOutlinedButton
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.HomeTodoListItem
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.RideSelector
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.ActivityAddDestination
import com.ramcosta.composedestinations.generated.destinations.BikeAddDestination
import com.ramcosta.composedestinations.generated.destinations.ChecklistDestination
import com.ramcosta.composedestinations.generated.destinations.HomeDestination
import com.ramcosta.composedestinations.generated.destinations.ListRideHistoryDestination
import com.ramcosta.composedestinations.generated.destinations.ManageActivitiesDestination
import com.ramcosta.composedestinations.generated.destinations.ManageBikesDestination
import com.ramcosta.composedestinations.generated.destinations.ManageComponentsDestination
import com.ramcosta.composedestinations.generated.destinations.PrepareForRideDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalFoundationApi::class, ExperimentalTime::class)
@Destination<RootGraph>(start = true)
@Composable
fun Home(
    homeViewModel: HomeViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle("") // Home doesn't have a title

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        HomeDestination,
        destinationsNavigator,
        windowSizeClass,
        myFloatingActionButton = KJsAction(
            enabled = true,
            imageVector = Icons.Default.Done,
            contentDescription = stringResource(R.string.btn_text_add_activity),
            onClick = {
                destinationsNavigator.navigate(ActivityAddDestination)
            }
        ),
        headline = "",
        advancedFunctionality = advancedFunctionality
    ) {
        val activitiesByBikes: List<ActivityWithBikeAndRideData> by homeViewModel
            .observeActivityWithBikeData.collectAsStateWithLifecycle(
                initialValue = emptyList()
            )
        val activitiesToBeShownHere = activitiesByBikes.filter { activityByBike ->
            activityByBike.activityIsCompleted == false && activityByBike.rideIsCompleted != true
        }
        var showDialog by remember { mutableStateOf(false) }

        val rides: List<Ride> by homeViewModel.observeRides.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val ridesInPreparation by homeViewModel.observeRidesInPrepararion.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )
        val bikes by homeViewModel.observeBikes.collectAsStateWithLifecycle(
            initialValue = emptyList()
        )

        Column(
            modifier = Modifier
                .fillMaxWidth()
        ) {
            Column(
                modifier = Modifier.weight(0.6f)
            ) {
                AnimatedVisibility(
                    visible = !advancedFunctionality || activitiesToBeShownHere.isNullOrEmpty()
                ) {
                    Column(
                        modifier = Modifier.padding(8.dp, 0.dp)
                    ) {
                        Text(
                            text = stringResource(R.string.text_welcome_to_jan_kev_s_bike_mechanical_disaster_prevention),
                            style = MaterialTheme.typography.bodyLarge.copy(
                                lineBreak = LineBreak.Paragraph
                            )
                        )
                        DefaultSpacer()
                        Text(
                            text = stringResource(R.string.text_this_is_a_tool_to_help_you_check_log_and_manage_critical_maintenance_tasks_on_your_bike),
                            style = MaterialTheme.typography.bodyLarge.copy(
                                lineBreak = LineBreak.Paragraph
                            )
                        )
                    }
                }
                AnimatedVisibility(
                    visible = advancedFunctionality && activitiesToBeShownHere.isNotEmpty()
                ) {
                    Text(
                        modifier = Modifier.padding(8.dp, 0.dp),
                        text = stringResource(R.string.text_here_are_your_top_activities_scroll_to_see_more_or_go_back_to_a_specific_ride_preparation_checklist),
                        style = MaterialTheme.typography.bodyLarge.copy(
                            lineBreak = LineBreak.Paragraph
                        )
                    )
                }
                DefaultSpacer()

                AnimatedVisibility(
                    visible = activitiesToBeShownHere.isNotEmpty()
                ) {
                    LazyColumn(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(8.dp, 0.dp)
                    ) {
                        stickyHeader {
                            if (windowSizeClass.heightSizeClass != WindowHeightSizeClass.Compact) {
                                Text(
                                    modifier = Modifier
                                        .fillMaxWidth()
                                        .background(MaterialTheme.colorScheme.primaryContainer)
                                        .padding(8.dp),
                                    text = "${stringResource(R.string.hdr_todos)} - unfinished activities"
                                )
                            }
                        }

                        items(
                            items = activitiesToBeShownHere,
                            key = { it.activityUid }) { activityByBike ->
                            val activity = Activity(
                                title = activityByBike.activityTitle,
                                description = activityByBike.activityDescription,
                                isCompleted = activityByBike.activityIsCompleted,
                                rideUid = activityByBike.rideUid,
                                isEBikeSpecific = activityByBike.isEBikeSpecific,
                                rideLevel = activityByBike.activityRideLevel,
                                createdInstant = activityByBike.activityCreatedInstant,
                                dueDate = activityByBike.activityDueDate,
                                bikeUid = activityByBike.bikeUid ?: 0,
                                doneInstant = activityByBike.activityDoneDateInstant,
                                componentUid = activityByBike.activityComponentUid,
                                uid = activityByBike.activityUid
                            )
                            HomeTodoListItem(
                                activity = activityByBike,
                                destinationsNavigator = destinationsNavigator,
                                onCheckboxCallback = { result ->
                                    homeViewModel.updateActivity(
                                        activity = activity.copy(
                                            isCompleted = result,
                                            doneInstant = if (result) {
                                                Clock.System.now()
                                            } else {
                                                null
                                            }
                                        )
                                    )
                                    // popup in case this has a component attached
                                    if (activityByBike.activityComponentUid != null) {
                                        // update component
                                        homeViewModel.updateAttachedComponent(
                                            componentUid = activityByBike.activityComponentUid,
                                            bikeUid = activityByBike.bikeUid
                                        )
                                        if (result) {
                                            // show UI for component wearlevel
                                            showDialog = true
                                        }
                                    }
                                },
                            )
                            if (showDialog) {
                                ComponentWearLevelDialog(
                                    onDismissRequest = {
                                        showDialog = false
                                    },
                                    onConfirmation = { wearLevel ->
                                        showDialog = false
                                        // write stuff
                                        if (activityByBike.activityComponentUid != null) {
                                            homeViewModel.logComponentWearLevel(
                                                componentUid = activityByBike.activityComponentUid,
                                                wearLevel = wearLevel
                                            )
                                        }
                                    },
                                    componentName = homeViewModel.getComponentNameForUid(
                                        activityByBike.activityComponentUid
                                    )
                                )
                            }
                        }
                    }
                }
            }

            DefaultSpacer()

            // buttons
            Column(
                modifier = Modifier
                    .height(IntrinsicSize.Min)
                    .padding(8.dp, 0.dp)
            ) {
                if (ridesInPreparation.isNotEmpty()) {
                    // we have an open ride prep
                    if (ridesInPreparation.size == 1) {
                        val rideUid = ridesInPreparation.first().uid
                        FullWidthButton(onClick = {
                            destinationsNavigator.navigate(ChecklistDestination(rideUid))
                        }) {
                            Text(text = stringResource(R.string.btn_lbl_back_to_ride_preparation))
                        }
                    } else {
                        RideSelector(
                            onRideSelected = { ride ->
                                val rideUid = ride?.uid
                                if (rideUid != null && rideUid > 0) {
                                    destinationsNavigator.navigate(ChecklistDestination(rideUid))
                                }
                            },
                            rides = ridesInPreparation
                        )
                    }
                    DefaultSpacer()
                }
                if (ridesInPreparation.isNotEmpty()) {
                    FullWidthOutlinedButton(onClick = {
                        destinationsNavigator.navigate(PrepareForRideDestination)
                    }) {
                        Text(text = stringResource(R.string.btn_desc_prepare_for_a_ride))
                    }
                } else {
                    FullWidthButton(onClick = {
                        destinationsNavigator.navigate(PrepareForRideDestination)
                    }) {
                        Text(text = stringResource(R.string.btn_desc_prepare_for_a_ride))
                    }
                }
                DefaultSpacer()

                AnimatedVisibility(
                    visible = windowSizeClass.widthSizeClass != WindowWidthSizeClass.Expanded
                ) {
                    FlowRow(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.SpaceBetween
                    ) {
                        if (bikes.isNotEmpty()) {
                            OutlinedButton(
                                onClick = {
                                    destinationsNavigator.navigate(ManageBikesDestination)
                                },
                                modifier = Modifier.padding(0.dp, 8.dp)
                            ) {
                                Text(text = stringResource(R.string.btn_text_manage_bikes))
                            }
                        } else {
                            OutlinedButton(
                                onClick = {
                                    destinationsNavigator.navigate(BikeAddDestination)
                                },
                                modifier = Modifier.padding(0.dp, 8.dp)
                            ) {
                                Text(text = stringResource(R.string.btn_text_add_bike))
                            }
                        }

                        OutlinedButton(
                            onClick = {
                                destinationsNavigator.navigate(ListRideHistoryDestination)
                            },
                            modifier = Modifier.padding(0.dp, 8.dp)
                        ) {
                            Text(text = stringResource(R.string.btn_text_ride_history))
                        }

                        if (windowSizeClass.widthSizeClass != WindowWidthSizeClass.Compact) {
                            OutlinedButton(
                                onClick = {
                                    destinationsNavigator.navigate(ManageActivitiesDestination)
                                },
                                modifier = Modifier.padding(0.dp, 8.dp)
                            ) {
                                Text(text = stringResource(R.string.btn_text_manage_activities))
                            }

                            if (advancedFunctionality) {
                                OutlinedButton(
                                    onClick = {
                                        destinationsNavigator.navigate(ManageComponentsDestination())
                                    },
                                    modifier = Modifier.padding(0.dp, 8.dp)
                                ) {
                                    Text(text = stringResource(R.string.btn_text_manage_components))
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}
