package com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations

import android.content.Context
import android.util.Log.d
import android.util.Log.w
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.StravaSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.network.strava.StravaApiManager
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import net.openid.appauth.AuthorizationException
import net.openid.appauth.AuthorizationResponse
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.OkHttpClient
import okhttp3.Request.Builder
import okhttp3.RequestBody.Companion.toRequestBody
import javax.inject.Inject
import kotlin.math.roundToInt

private const val TAG = "syncStravaVM"

@HiltViewModel
class SyncWithStravaViewModel @Inject constructor(
    val repository: KJsRepository,
    val userPreferencesManager: UserPreferencesManager,
    val stravaApiManager: StravaApiManager
) : ViewModel() {

    private val clientId = "52749"

    private val _listOfBikesOnStrava: MutableStateFlow<List<StravaSummaryGear>> = MutableStateFlow(
        emptyList()
    )
    val listOfBikesOnStrava: StateFlow<List<StravaSummaryGear>> = _listOfBikesOnStrava

    val observeBikes = repository.observeBikes

    val observeStravaSummaryGear = repository.observeStravaSummaryGear

    private val _mapOfStravaBikeConnections: MutableStateFlow<Map<String, Bike>> = MutableStateFlow(
        emptyMap()
    )
    val mapOfStravaBikeConnections: StateFlow<Map<String, Bike>> = _mapOfStravaBikeConnections

    val isStravaConnected: StateFlow<Boolean> = flow { // TODO this looks clunky AF!
        while (true) {
            delay(500)
            val isConnected = stravaApiManager.isConnected()
            emit(isConnected)
        }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )
    val isAllowedToReadAllProfileData: StateFlow<Boolean> = flow { // TODO this looks clunky AF!
        while (true) {
            delay(500)
            emit(stravaApiManager.isAllowedToReadAllProfileData())
        }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    init {
        // we do not want to retrieve bikes from Strava every time, so...
        viewModelScope.launch {
            // build the list
            _listOfBikesOnStrava.value = repository.getAllStravaSummaryGear()
            // build the map
            val listOfMappedBikes = repository.getAllBikes().filter { bike ->
                !bike.stravaGearId.isNullOrEmpty()
            }
            val tempMapOfStravaBikeConnections: MutableMap<String, Bike> = mutableMapOf()
            listOfMappedBikes.forEach { bike ->
                if (!bike.stravaGearId.isNullOrEmpty()) {
                    tempMapOfStravaBikeConnections.put(bike.stravaGearId, bike)
                }
            }
            _mapOfStravaBikeConnections.value = tempMapOfStravaBikeConnections
        }
    }

    fun disconnectStrava() {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                d(TAG, "Trying to end session...")
                val client = OkHttpClient()
                val mediaType = "application/json; charset=utf-8".toMediaTypeOrNull()
                val requestBody = "{}".toRequestBody(mediaType)

                // Build and execute the POST request
                val request = Builder()
                    .url("https://www.strava.com/oauth/deauthorize")
                    .post(requestBody)
                    .build()

                client.newCall(request).execute()
            }
        }
        stravaApiManager.disconnect()
    }

    fun useCode(context: Context, code: String) {
        viewModelScope.launch {
            stravaApiManager.useCode(context, code, clientId) { isConnected ->
                d(TAG, "Strava authState connected $isConnected")
            }
        }
    }

    fun applyAllowedScopes(scopeString: String) {
        viewModelScope.launch {
            stravaApiManager.applyAllowedScopes(scopeString)
        }
    }

    fun connect(context: Context) {
        viewModelScope.launch {
            stravaApiManager.connect(context, clientId)
        }
    }

    fun updateAuthState(
        authResponse: AuthorizationResponse?,
        authException: AuthorizationException?
    ) {
        stravaApiManager.updateAuthState(authResponse, authException)
    }

    fun retrieveBikesFromStrava(context: Context) {
        viewModelScope.launch {
            if (isStravaConnected.value) {
                stravaApiManager.retrieveLoggedInAthlete(
                    context = context,
                    onResult = { stravaDetailedAthlete ->
                        if (stravaDetailedAthlete != null) {
                            val tempBikes = stravaDetailedAthlete.bikes
                            // put them into a list, and find existing links
                            val templist = listOfBikesOnStrava.value.toMutableList()
                            tempBikes?.forEach { gear ->
                                templist.add(gear)
                                if (gear.id != null) {
                                    viewModelScope.launch {
                                        val bike = repository.getBikeForStravaGearId(gear.id)
                                        if (bike != null) {
                                            val tempStravaBikeConnections =
                                                mapOfStravaBikeConnections.value.toMutableMap()
                                            tempStravaBikeConnections[gear.id] = bike
                                            _mapOfStravaBikeConnections.value =
                                                tempStravaBikeConnections
                                        }
                                        // store or update
                                        val tempGear = repository.getStravaSummaryGearById(gear.id)
                                        if (tempGear != null) {
                                            repository.updateStravaSummaryGear(gear)
                                        } else {
                                            repository.insertStravaSummaryGear(gear)
                                        }
                                    }
                                }
                            }
                            _listOfBikesOnStrava.value = templist
                        }
                    },
                    onError = { code ->
                        w(TAG, "Error $code")
                    }
                )
            }
        }
    }

    fun onBikeSelected(gearId: String, bike: Bike) {
        // update map
        val tempMap = mapOfStravaBikeConnections.value.toMutableMap()
        tempMap.remove(gearId)
        tempMap[gearId] = bike
        _mapOfStravaBikeConnections.value = tempMap
        viewModelScope.launch {
            // update _other_ bikes to remove this gear_id
            val otherBike = repository.getBikeForStravaGearId(gearId)
            if (otherBike != null) {
                val changedBike = otherBike.copy(
                    stravaGearId = ""
                )
                repository.updateBike(changedBike)
            }
            // update bike with strava_gear_id
            val changedSelectedBike = bike.copy(
                stravaGearId = gearId,
            )
            repository.updateBike(changedSelectedBike)
        }
    }

    fun onBikeInserted(gear: StravaSummaryGear, bike: Bike) {
        val gearId = gear.id!!
        // update map
        val tempMap = mapOfStravaBikeConnections.value.toMutableMap()
        tempMap.remove(gearId)
        tempMap[gearId] = bike
        _mapOfStravaBikeConnections.value = tempMap
        viewModelScope.launch {
            // update _other_ bikes to remove this gear_id
            val otherBike = repository.getBikeForStravaGearId(gearId)
            if (otherBike != null) {
                val changedBike = otherBike.copy(
                    stravaGearId = ""
                )
                repository.updateBike(changedBike)
            }
            // update bike with strava_gear_id
            val changedSelectedBike = bike.copy(
                stravaGearId = gearId,
            )
            val bikeUid = repository.insertBike(changedSelectedBike)
            // update the map so we have the bikeUid
            // it is needed to retrieve last ride date
            val temp1 = mapOfStravaBikeConnections.value[gearId]
            if (temp1 != null) {
                val temp2 = temp1.copy(
                    uid = bikeUid
                )
                val temp3 = mapOfStravaBikeConnections.value.toMutableMap()
                temp3[gearId] = temp2
                _mapOfStravaBikeConnections.value = temp3
            }
        }
    }

    fun updateDistancesFromStrava(
        context: Context,
        onResult: () -> Unit,
        onError: (Int) -> Unit
    ) {
        viewModelScope.launch {
            val bikes = repository.getAllBikes()
            bikes.forEach { bike ->
                updateMileageOnBikeAndComponents(
                    context = context,
                    bike = bike,
                    onResult = {
                        // TODO
                    },
                    onError = { code ->
                        w(TAG, "Error $code getting distances")
                        onError(code)
                    }
                )
            }
            // done launching for all bikes
            onResult()
        }
    }

    private fun updateMileageOnBikeAndComponents(
        context: Context,
        bike: Bike,
        onResult: () -> Unit,
        onError: (Int) -> Unit
    ) {
        if (!bike.stravaGearId.isNullOrEmpty()) {
            viewModelScope.launch {
                stravaApiManager.retrieveGear(
                    context = context,
                    gearId = bike.stravaGearId,
                    onResult = { gear ->
                        viewModelScope.launch {
                            val beforeMileage = bike.mileage
                            val newDistance = gear.distance
                            if (newDistance != null) {
                                val newRealDistance = (gear.distance / 1000).roundToInt()
                                if (beforeMileage < newRealDistance) {
                                    val difference = newRealDistance - beforeMileage
                                    // add this distance to every component attached to this bike
                                    val allComponentsForBike =
                                        repository.getAllComponentsForBike(bike.uid)
                                    allComponentsForBike.forEach { component ->
                                        val newMileage =
                                            difference + (component.currentMileage ?: 0)
                                        val updatedComponent = component.copy(
                                            currentMileage = newMileage
                                        )
                                        repository.updateComponent(updatedComponent)
                                    }
                                }
                                // update the bike itself
                                val updatedBike = bike.copy(
                                    mileage = newRealDistance
                                )
                                repository.updateBike(updatedBike)
                            }
                            onResult()
                        }
                    },
                    onError = { code ->
                        w(TAG, "Error retrieving gear ${bike.stravaGearId}")
                        onError(code)
                    }
                )
            }
        } else {
            w(TAG, "Bike ${bike.name} does not have a Strava Gear ID")
            onError(404)
        }
    }
}