package com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations

import android.content.Context
import android.util.Log
import android.util.Log.d
import android.util.Log.w
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.IntervalsSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.network.intervals.IntervalsApiManager
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import net.openid.appauth.AuthorizationException
import net.openid.appauth.AuthorizationResponse
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.OkHttpClient
import okhttp3.Request.Builder
import javax.inject.Inject
import kotlin.math.roundToInt

private const val TAG = "syncIntervalsVM"

// see https://forum.intervals.icu/t/intervals-icu-oauth-support/2759 for help

@HiltViewModel
class SyncWithIntervalsViewModel @Inject constructor(
    val repository: KJsRepository,
    val userPreferencesManager: UserPreferencesManager,
    val intervalsApiManager: IntervalsApiManager
) : ViewModel() {

    private val clientId = "98"

    private val metricsFactor = 1.609344f

    private val _listOfBikesOnIntervals: MutableStateFlow<List<IntervalsSummaryGear>> =
        MutableStateFlow(
            emptyList()
        )
    val listOfBikesOnIntervals: StateFlow<List<IntervalsSummaryGear>> = _listOfBikesOnIntervals

    val observeBikes = repository.observeBikes

    val observeIntervalsSummaryGear = repository.observeIntervalsSummaryGear

    private val _mapOfIntervalsBikeConnections: MutableStateFlow<Map<String, Bike>> =
        MutableStateFlow(
            emptyMap()
        )
    val mapOfIntervalsBikeConnections: StateFlow<Map<String, Bike>> = _mapOfIntervalsBikeConnections

    val isIntervalsConnected: StateFlow<Boolean> = flow { // TODO this looks clunky AF!
        while (true) {
            delay(500)
            val isConnected = intervalsApiManager.isConnected()
            emit(isConnected)
        }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )
    val isAllowedToReadAthleteSettings: StateFlow<Boolean> = flow { // TODO this looks clunky AF!
        while (true) {
            delay(500)
            emit(intervalsApiManager.isAllowedToReadAthleteSettings())
        }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    val metricsPreferenceLocal = userPreferencesManager.distanceMeasure().stateIn(
        viewModelScope,
        SharingStarted.Eagerly,
        DistanceMeasure.KM
    )
    val metricsPreferenceInterval = userPreferencesManager.distanceMeasureIntervals().stateIn(
        viewModelScope,
        SharingStarted.Eagerly,
        DistanceMeasure.KM
    )

    init {
        // we do not want to retrieve bikes from Intervals every time, so...
        viewModelScope.launch {
            // build the list
            _listOfBikesOnIntervals.value = repository.getAllIntervalsSummaryGear()
            // build the map
            val listOfMappedBikes = repository.getAllBikes().filter { bike ->
                !bike.intervalsGearId.isNullOrEmpty()
            }
            val tempMapOfIntervalsBikeConnections: MutableMap<String, Bike> = mutableMapOf()
            listOfMappedBikes.forEach { bike ->
                if (!bike.intervalsGearId.isNullOrEmpty()) {
                    tempMapOfIntervalsBikeConnections.put(bike.intervalsGearId, bike)
                }
            }
            _mapOfIntervalsBikeConnections.value = tempMapOfIntervalsBikeConnections
        }
    }

    fun disconnect() {
        viewModelScope.launch {
            withContext(Dispatchers.IO) {
                d(TAG, "Trying to end session...")
                val client = OkHttpClient()
                val mediaType = "application/json; charset=utf-8".toMediaTypeOrNull()

                // Build and execute the POST request
                val request = Builder()
                    .url("https://intervals.icu/api/v1/disconnect-app")
                    .build()

                client.newCall(request).execute()
            }
        }
        intervalsApiManager.disconnect()
    }

    fun useCode(context: Context, code: String) {
        viewModelScope.launch {
            intervalsApiManager.useCode(context, code, clientId) { isConnected ->
                d(TAG, "Intervals.icu authState connected $isConnected")
            }
        }
    }

    fun applyAllowedScopes(scopeString: String) {
        viewModelScope.launch {
            intervalsApiManager.applyAllowedScopes(scopeString)
        }
    }

    fun connect(context: Context) {
        viewModelScope.launch {
            intervalsApiManager.connect(context, clientId)
        }
    }

    fun updateAuthState(
        authResponse: AuthorizationResponse?,
        authException: AuthorizationException?
    ) {
        intervalsApiManager.updateAuthState(authResponse, authException)
    }

    fun retrieveBikesFromIntervals(context: Context) {
        viewModelScope.launch {
            if (isIntervalsConnected.value) {
                intervalsApiManager.retrieveLoggedInAthlete(
                    context = context,
                    onResult = { intervalsAthlete ->
                        if (intervalsAthlete != null) {
                            // before we do anything else, let's see whether they prefer metric or not
                            val measurementPreferenceString = intervalsAthlete.measurementPreference
                            if (!measurementPreferenceString.isNullOrEmpty()) {
                                val metricsPreference =
                                    if (measurementPreferenceString == "meters") {
                                        DistanceMeasure.KM
                                    } else {
                                        DistanceMeasure.MI
                                    }
                                // and save that into the preferences
                                // yes, every time! They may change it
                                CoroutineScope(Dispatchers.Default).launch {
                                    userPreferencesManager.setDistanceMeasureIntervals(
                                        metricsPreference
                                    )
                                }
                            }
                            val tempBikes = intervalsAthlete.bikes
                            // put them into a list, and find existing links
                            val templist = listOfBikesOnIntervals.value.toMutableList()
                            tempBikes?.forEach { gear ->
                                var fixedGear = gear
                                // new gear may have a different metric! Fix that rirst
                                if (metricsPreferenceLocal != metricsPreferenceInterval) {
                                    // yup, needs fixing
                                    val distanceFromGear = fixedGear.distance
                                    if (metricsPreferenceLocal.value == DistanceMeasure.KM && metricsPreferenceInterval.value == DistanceMeasure.MI) {
                                        // mi -> km, multiply
                                        fixedGear = fixedGear.copy(
                                            distance = distanceFromGear?.times(metricsFactor)
                                        )
                                    } else if (metricsPreferenceLocal.value == DistanceMeasure.MI && metricsPreferenceInterval.value == DistanceMeasure.KM) {
                                        // km -> mi, divide
                                        fixedGear = fixedGear.copy(
                                            distance = distanceFromGear?.div(metricsFactor)
                                        )
                                    } else {
                                        // Well... this should NEVER happen!
                                        // but I do not want to crash the app, either...
                                        // people will report when metrics are off
                                    }
                                }
                                // now save it
                                templist.add(fixedGear)
                                if (fixedGear.id != null) {
                                    viewModelScope.launch {
                                        val bike =
                                            repository.getBikeForIntervalsGearId(fixedGear.id)
                                        if (bike != null) {
                                            val tempIntervalsBikeConnections =
                                                mapOfIntervalsBikeConnections.value.toMutableMap()
                                            tempIntervalsBikeConnections[fixedGear.id] = bike
                                            _mapOfIntervalsBikeConnections.value =
                                                tempIntervalsBikeConnections
                                        }
                                        // store or update
                                        val tempGear =
                                            repository.getIntervalsSummaryGearById(fixedGear.id)
                                        if (tempGear != null) {
                                            repository.updateIntervalsSummaryGear(fixedGear)
                                        } else {
                                            repository.insertIntervalsSummaryGear(fixedGear)
                                        }
                                    }
                                }
                            }
                            _listOfBikesOnIntervals.value = templist
                        }
                    },
                    onError = { code ->
                        w(TAG, "Error $code")
                    }
                )
            }
        }
    }

    fun onBikeSelected(gearId: String, bike: Bike) {
        // update map
        val tempMap = mapOfIntervalsBikeConnections.value.toMutableMap()
        tempMap.remove(gearId)
        tempMap[gearId] = bike
        _mapOfIntervalsBikeConnections.value = tempMap
        viewModelScope.launch {
            // update _other_ bikes to remove this gear_id
            val otherBike = repository.getBikeForIntervalsGearId(gearId)
            if (otherBike != null) {
                val changedBike = otherBike.copy(
                    intervalsGearId = ""
                )
                repository.updateBike(changedBike)
            }
            // update bike with strava_gear_id
            val changedSelectedBike = bike.copy(
                intervalsGearId = gearId,
            )
            repository.updateBike(changedSelectedBike)
        }
    }

    fun onBikeInserted(gear: IntervalsSummaryGear, bike: Bike) {
        val gearId = gear.id!!
        // update map
        val tempMap = mapOfIntervalsBikeConnections.value.toMutableMap()
        tempMap.remove(gearId)
        tempMap[gearId] = bike
        _mapOfIntervalsBikeConnections.value = tempMap
        viewModelScope.launch {
            // update _other_ bikes to remove this gear_id
            val otherBike = repository.getBikeForIntervalsGearId(gearId)
            if (otherBike != null) {
                val changedBike = otherBike.copy(
                    intervalsGearId = ""
                )
                repository.updateBike(changedBike)
            }
            // update bike with strava_gear_id
            val changedSelectedBike = bike.copy(
                intervalsGearId = gearId,
            )
            val bikeUid = repository.insertBike(changedSelectedBike)
            // update the map so we have the bikeUid
            // it is needed to retrieve last ride date
            val temp1 = mapOfIntervalsBikeConnections.value[gearId]
            if (temp1 != null) {
                val temp2 = temp1.copy(
                    uid = bikeUid
                )
                val temp3 = mapOfIntervalsBikeConnections.value.toMutableMap()
                temp3[gearId] = temp2
                _mapOfIntervalsBikeConnections.value = temp3
            }
        }
    }

    fun updateDistancesFromIntervals(
        context: Context,
        onResult: () -> Unit,
        onError: (Int) -> Unit
    ) {
        viewModelScope.launch {
            val bikes = repository.getAllBikes()
            updateMileageOnBikeAndComponents(
                context = context,
                bikes = bikes,
                onError = { code ->
                    w(TAG, "Error $code getting distances")
                    onError(code)
                }
            )
            // done launching the process for each bike
            onResult()
        }
    }

    private fun updateMileageOnBikeAndComponents(
        context: Context,
        bikes: List<Bike>,
        onError: (Int) -> Unit
    ) {
        if (bikes.isNotEmpty()) {
            viewModelScope.launch {
                intervalsApiManager.retrieveGear(
                    context = context,
                    onResult = { gearList ->
                        gearList.forEach { gear ->
                            var fixedGear = gear
                            // new gear may have a different metric! Fix that rirst
                            Log.d("SWIVM", "Conversion? ${metricsPreferenceLocal.value}/${metricsPreferenceInterval.value} ${fixedGear.distance}")
                            if (metricsPreferenceLocal.value != metricsPreferenceInterval.value) {
                                // yup, needs fixing
                                val distanceFromGear = fixedGear.distance
                                if (metricsPreferenceLocal.value == DistanceMeasure.KM && metricsPreferenceInterval.value == DistanceMeasure.MI) {
                                    // mi -> km, multiply
                                    fixedGear = fixedGear.copy(
                                        distance = distanceFromGear?.times(metricsFactor)
                                    )
                                } else if (metricsPreferenceLocal.value == DistanceMeasure.MI && metricsPreferenceInterval.value == DistanceMeasure.KM) {
                                    // km -> mi, divide
                                    fixedGear = fixedGear.copy(
                                        distance = distanceFromGear?.div(metricsFactor)
                                    )
                                } else {
                                    // Well... this should NEVER happen!
                                    // but I do not want to crash the app, either...
                                    // people will report when metrics are off
                                }
                            }
                            // now use it
                            viewModelScope.launch {
                                if (fixedGear.id != null) {
                                    val bike = repository.getBikeForIntervalsGearId(fixedGear.id)
                                    if (bike != null) {
                                        val beforeMileage = bike.mileage
                                        val newDistance = fixedGear.distance
                                        if (newDistance != null) {
                                            val newRealDistance =
                                                (fixedGear.distance / 1000).roundToInt()
                                            if (beforeMileage < newRealDistance) {
                                                val difference = newRealDistance - beforeMileage
                                                // add this distance to every component attached to this bike
                                                val allComponentsForBike =
                                                    repository.getAllComponentsForBike(bike.uid)
                                                allComponentsForBike.forEach { component ->
                                                    val newMileage =
                                                        difference + (component.currentMileage ?: 0)
                                                    val updatedComponent = component.copy(
                                                        currentMileage = newMileage
                                                    )
                                                    repository.updateComponent(updatedComponent)
                                                }
                                            }
                                            // update the bike itself
                                            val updatedBike = bike.copy(
                                                mileage = newRealDistance
                                            )
                                            repository.updateBike(updatedBike)
                                        }
                                    }
                                }
                            }
                        }
                    },
                    onError = { code ->
                        w(TAG, "Error retrieving gear $code")
                        onError(code)
                    }
                )
            }
        } else {
            w(TAG, "There are no bikes to retrieve")
            onError(404)
        }
    }
}
