package com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Download
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Text
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.IntervalsSummaryGear
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsGlobalScaffoldViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.KJsResponsiveNavigation
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.BikeSelectorWithTextButtonAndAddBikeOption
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.DefaultSpacer
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents.ImportBikeDataDialog
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.KJsAction
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootGraph
import com.ramcosta.composedestinations.generated.destinations.HomeDestination
import com.ramcosta.composedestinations.generated.destinations.SyncWithStravaDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.math.roundToInt
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

@OptIn(ExperimentalTime::class)
@Destination<RootGraph>
@Composable
fun SyncWithIntervals(
    syncWithIntervalsViewModel: SyncWithIntervalsViewModel = hiltViewModel(),
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel,
    destinationsNavigator: DestinationsNavigator,
    windowSizeClass: WindowSizeClass
) {
    kJsGlobalScaffoldViewModel.setDestinationTitle(stringResource(R.string.hdr_sync_with_strava)) // Home doesn't have a title

    val context = LocalContext.current

    val bikesOnIntervals by syncWithIntervalsViewModel.observeIntervalsSummaryGear.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )

    val bikes: List<Bike> by syncWithIntervalsViewModel.observeBikes.collectAsStateWithLifecycle(
        initialValue = emptyList()
    )

    val mapOfIntervalsBikeConnections by syncWithIntervalsViewModel.mapOfIntervalsBikeConnections.collectAsStateWithLifecycle(
        initialValue = emptyMap()
    )

    val isIntervalsConnected by syncWithIntervalsViewModel.isIntervalsConnected.collectAsStateWithLifecycle()
    val isAllowedToReadAthleteSettings by syncWithIntervalsViewModel.isAllowedToReadAthleteSettings.collectAsStateWithLifecycle()

    var newBikeDialogOpen: Boolean by remember { mutableStateOf(false) }
    var newGear: IntervalsSummaryGear? by remember { mutableStateOf(null) }
    var newBike: Bike? by remember { mutableStateOf(null) }

    val advancedFunctionality by kJsGlobalScaffoldViewModel.advancedFunctionality.collectAsStateWithLifecycle()

    KJsResponsiveNavigation(
        SyncWithStravaDestination,
        destinationsNavigator,
        windowSizeClass,
        myActions = listOf(
            KJsAction(
                imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                contentDescription = stringResource(R.string.btn_desc_back),
                onClick = {
                    destinationsNavigator.popBackStack(
                        HomeDestination,
                        inclusive = false
                    )
                }
            )
        ),
        myFloatingActionButton = KJsAction(
            enabled = isAllowedToReadAthleteSettings,
            imageVector = Icons.Default.Download,
            contentDescription = stringResource(R.string.btn_text_retrieve_bikes_from_intervals),
            onClick = {
                syncWithIntervalsViewModel.retrieveBikesFromIntervals(context)
            }
        ),
        headline = "",
        advancedFunctionality = advancedFunctionality
    ) {
        Column(
            modifier = Modifier
                .verticalScroll(rememberScrollState())
        ) {
            if (isIntervalsConnected) {
                if (bikesOnIntervals.isNotEmpty()) {
                    Text(text = stringResource(R.string.bikes_retrieved_from_intervals) + ": ${bikesOnIntervals.size}")
                    DefaultSpacer()
                    HorizontalDivider()
                    DefaultSpacer()
                    LazyColumn(
                        modifier = Modifier.height(256.dp)
                    ) {
                        item {
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween,
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Text(text = stringResource(R.string.intervals))
                                Text(text = stringResource(R.string.local_this_app))
                            }
                        }
                        items(items = bikesOnIntervals, key = { it.id!! }) { gear ->
                            Row(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .height(44.dp),
                                horizontalArrangement = Arrangement.SpaceBetween,
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Text(
                                    text = "${gear.name} (${gear.id})",
                                    overflow = TextOverflow.Ellipsis
                                )
                                BikeSelectorWithTextButtonAndAddBikeOption(
                                    currentBike = mapOfIntervalsBikeConnections[gear.id],
                                    bikes = bikes,
                                    displayLabel = false,
                                    addFromIntegrationLabelString = context.getString(R.string.dropdown_menu_item_bike_import_from_intervals),
                                    onBikeSelected = { bike ->
                                        if (bike != null && gear.id != null) {
                                            if (bike.name == context.getString(R.string.dropdown_menu_item_bike_import_from_intervals)) {
                                                // a bike to be imported
                                                newGear = gear
                                                newBike = bike
                                                newBikeDialogOpen = true
                                            } else {
                                                // an existing bike being linked
                                                syncWithIntervalsViewModel.onBikeSelected(
                                                    gearId = gear.id,
                                                    bike = bike,
                                                )
                                            }
                                        }
                                    }
                                )
                            }
                        }
                    }
                    // logo
                    DefaultSpacer()
                    HorizontalDivider()
                    DefaultSpacer()
                    IntervalsAttributionTextAndLogo()
                    if (newBikeDialogOpen) {
                        ImportBikeDataDialog(
                            onDismissRequest = { },
                            onConfirmation = { isEbike, acquisitionDate ->
                                newBikeDialogOpen = false
                                if (newGear != null && newBike != null) {
                                    val importedBike = newBike!!.copy(
                                        name = newGear!!.name
                                            ?: context.getString(R.string.default_name_from_intervals),
                                        mileage = ((newGear!!.distance
                                            ?: 0.0f) / 1000).roundToInt(),
                                        stravaGearId = newGear!!.id,
                                        isElectric = isEbike,
                                        buildDate = Instant.fromEpochMilliseconds(acquisitionDate)
                                            .toLocalDateTime(
                                                TimeZone.currentSystemDefault()
                                            ).date
                                    )
                                    syncWithIntervalsViewModel.onBikeInserted(
                                        gear = newGear!!,
                                        bike = importedBike,
                                    )
                                }
                            }
                        )
                    }
                } else {
                    Text(text = stringResource(R.string.text_use_the_button_at_the_bottom_to_retrieve_bikes_from_your_intervals_account))
                }
                // note for those who are not trusting the app
                if (!isAllowedToReadAthleteSettings) {
                    DefaultSpacer()
                    HorizontalDivider()
                    DefaultSpacer()
                    Text(stringResource(R.string.text_unable_to_retrieve_bikes_from_strava))
                    DefaultSpacer()
                    Text(text = stringResource(R.string.text_you_must_set_the_view_your_complete_strava_profile_permission_when_you_authorize_the_app))
                    DefaultSpacer()
                    Text(text = stringResource(R.string.text_to_fix_this_disconnect_then_connect_again_and_leave_the_permission_enabled))
                }
            } else {
                Text(text = stringResource(R.string.unknown_status))
            }
        }
    }
}