package com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers

import kotlinx.datetime.DateTimeUnit
import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalTime
import kotlin.time.Instant
import kotlinx.datetime.TimeZone
import kotlinx.datetime.format.FormatStringsInDatetimeFormats
import kotlinx.datetime.minus
import kotlinx.datetime.toLocalDateTime
import kotlin.time.Clock
import kotlin.time.Duration
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
fun convertMillisToDate(millis: Long): String {
    return Instant.fromEpochMilliseconds(millis)
        .toLocalDateTime(TimeZone.currentSystemDefault()).date.toString()
}

@OptIn(ExperimentalTime::class)
fun convertMillisToDateAndTime(millis: Long): String {
    return Instant.fromEpochMilliseconds(millis)
        .toLocalDateTime(TimeZone.currentSystemDefault()).toString()
}

@OptIn(ExperimentalTime::class, FormatStringsInDatetimeFormats::class)
fun formatDateAndTimeSensibly(instantISO8601: String): String {
    val instant = Instant.parse(instantISO8601)

    return formatDateAndTimeSensibly(instant)
}

@OptIn(ExperimentalTime::class, FormatStringsInDatetimeFormats::class)
fun formatDateAndTimeSensibly(instant: Instant): String {
    // is it today?
    val localDateTime = instant.toLocalDateTime(TimeZone.currentSystemDefault())
    val localDate = localDateTime.date
    val today = Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault()).date
    if (localDate == today) {
        val localTime = LocalTime(localDateTime.time.hour, localDateTime.minute, localDateTime.second)
        return "Today at localTime"
    }
    // or yesterday?
    val yesterday = today.minus(1, DateTimeUnit.DAY)
    if (localDate == yesterday) {
        return "Yesterday"
    }

    // otherwise
    return localDate.toString()
}

fun formatDateAndTimeDifferenceSensibly(startInstant: Instant, stopInstant: Instant): String {
    val difference = stopInstant.minus(startInstant)
    return formatDurationSensibly(difference)
}

fun formatDurationSensibly(duration: Duration): String {
    val durationInSeconds = duration.toLong(DurationUnit.SECONDS)
    if (durationInSeconds < 60) {
        return "less than a minute"
    } else if (durationInSeconds < (2 * 3600)) {
        return "${duration.inWholeMinutes} minutes"
    } else if (durationInSeconds < (2 * 86400)) {
        return "${duration.inWholeHours} hours"
    }
    return "${duration.inWholeDays} days"
}

fun convertPotentiallyEmptyStringToLocalDate(input: String?): LocalDate? {
    if (input.isNullOrEmpty()) {
        return null
    }
    return LocalDate.parse(input)
}

fun potentiallyEmptyLocalDateToString(localDate: LocalDate?): String? {
    if (localDate == null) {
        return null
    }
    return localDate.toString()
}