package com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents

import androidx.compose.foundation.clickable
import androidx.compose.material3.Checkbox
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.withStyle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.TemplateActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.TemplateActivityType
import com.exner.tools.jkbikemechanicaldisasterprevention.database.views.ActivityWithBikeAndRideData
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.RideLevel
import com.ramcosta.composedestinations.generated.destinations.ActivityEditDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import kotlin.time.Clock
import kotlinx.datetime.TimeZone
import kotlinx.datetime.todayIn
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Composable
fun TodoListItem(
    activity: ActivityWithBikeAndRideData,
    destinationsNavigator: DestinationsNavigator,
    onCheckboxCallback: (Boolean) -> Unit,
    suppressBikeBadge: Boolean = false,
    suppressDueDate: Boolean = false
) {
    Surface(
        modifier = Modifier
            .clickable {
                destinationsNavigator.navigate(
                    ActivityEditDestination(
                        activity.activityUid
                    )
                )
            },
    ) {
        ListItem(
            overlineContent = {
                if (activity.bikeName != null && !suppressBikeBadge) {
                    Text(text = activity.bikeName, color = MaterialTheme.colorScheme.tertiary)
                }
            },
            headlineContent = {
                if (activity.activityDueDate != null && !suppressDueDate) {
                    val colour = if (activity.activityDueDate > Clock.System.todayIn(TimeZone.currentSystemDefault())) {
                        Color.Red
                    } else {
                        MaterialTheme.colorScheme.onSurface
                    }
                    Text(
                        buildAnnotatedString {
                            withStyle(style = SpanStyle(color = colour)) {
                                append("${activity.activityDueDate}")
                            }
                            append(" - ${activity.activityTitle}")
                        }
                    )
                } else {
                    Text(text = activity.activityTitle)
                }
            },
            supportingContent = {
                Text(text = activity.activityDescription)
                // TODO add component information
            },
            trailingContent = {
                Checkbox(
                    checked = activity.activityIsCompleted,
                    onCheckedChange = {
                        onCheckboxCallback(!activity.activityIsCompleted)
                    }
                )
            }
        )
    }
}

@OptIn(ExperimentalTime::class)
@Composable
fun HomeTodoListItem(
    activity: ActivityWithBikeAndRideData,
    destinationsNavigator: DestinationsNavigator,
    onCheckboxCallback: (Boolean) -> Unit,
) {
    Surface(
        modifier = Modifier
            .clickable {
                destinationsNavigator.navigate(
                    ActivityEditDestination(
                        activity.activityUid
                    )
                )
            },
    ) {
        ListItem(
            overlineContent = {
                if (activity.bikeName != null) {
                    Text(text = activity.bikeName, color = MaterialTheme.colorScheme.tertiary)
                }
            },
            headlineContent = {
                if (activity.activityDueDate != null) {
                    val colour = if (activity.activityDueDate > Clock.System.todayIn(TimeZone.currentSystemDefault())) {
                        Color.Red
                    } else {
                        MaterialTheme.colorScheme.onSurface
                    }
                    Text(
                        buildAnnotatedString {
                            withStyle(style = SpanStyle(color = colour)) {
                                append("${activity.activityDueDate}")
                            }
                            append(" - ${activity.activityTitle}")
                        }
                    )
                } else {
                    Text(text = activity.activityTitle)
                }
            },
            trailingContent = {
                Checkbox(
                    checked = activity.activityIsCompleted,
                    onCheckedChange = {
                        onCheckboxCallback(!activity.activityIsCompleted)
                    }
                )
            }
        )
    }
}

@Composable
fun TransientTodoListItem(
    activity: ActivityWithBikeAndRideData,
    onCheckboxCallback: (Boolean) -> Unit,
    suppressDueDate: Boolean = false
) {
    val context = LocalContext.current
    val overlineText = RideLevel.getLabel(context, activity.activityRideLevel) + if (activity.isEBikeSpecific) {
        stringResource(
            R.string.dash_ebike_specific
        ) } else { "" }
    ListItem(
        overlineContent = {
            Text(text = overlineText, color = MaterialTheme.colorScheme.tertiary)
        },
        headlineContent = {
            val headline = if (activity.activityDueDate != null && !suppressDueDate) {
                "${activity.activityDueDate} - ${activity.activityTitle}"
            } else {
                activity.activityTitle
            }
            Text(text = headline)
        },
        supportingContent = {
            Text(text = activity.activityDescription)
        },
        trailingContent = {
            Checkbox(
                checked = activity.activityIsCompleted,
                onCheckedChange = {
                    onCheckboxCallback(!activity.activityIsCompleted)
                }
            )
        }
    )
}

@Composable
fun TemplateActivityListItem(
    templateActivity: TemplateActivity,
    onItemClick: (Long) -> Unit
) {
    val context = LocalContext.current
    val overlineContentText = RideLevel.getLabel(context, templateActivity.rideLevel) + if (templateActivity.isEBikeSpecific) { stringResource(
        R.string.dash_ebike_specific
    ) } else { "" } + if (templateActivity.typeOfTemplate == TemplateActivityType.BUILT_IN) { " - built-in" } else { "" }
    ListItem(
        modifier = Modifier.clickable { onItemClick(templateActivity.uid) },
        overlineContent = {
            Text(text = overlineContentText, color = MaterialTheme.colorScheme.tertiary)
        },
        headlineContent = {
            Text(text = templateActivity.title)
        },
        supportingContent = {
            Text(text = templateActivity.description)
        },
    )
}
