package com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.HelpOutline
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Surface
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.material3.TextFieldDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.core.text.isDigitsOnly
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.BackgroundWorkLog
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.Theme
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import kotlinx.datetime.TimeZone
import kotlinx.datetime.number
import kotlinx.datetime.toLocalDateTime

@Composable
fun DefaultTextFieldWithSpacer(
    modifier: Modifier = Modifier,
    value: String = "",
    label: String = "",
    placeholder: String? = null,
    onValueChange: (String) -> Unit,
) {
    val usablePlaceholder = placeholder ?: label

    OutlinedTextField(
        modifier = modifier
            .fillMaxWidth(),
        value = value,
        onValueChange = onValueChange,
        placeholder = { Text(text = usablePlaceholder) },
        label = { Text(text = label) },
        singleLine = true,
        keyboardOptions = KeyboardOptions.Default.copy(
            capitalization = KeyboardCapitalization.Sentences
        )
    )
    DefaultSpacer()
}

@Composable
fun DefaultNumberFieldWithSpacer(
    modifier: Modifier = Modifier,
    value: String = "",
    label: String = "",
    enabled: Boolean = true,
    placeholder: String? = null,
    onValueChange: (String) -> Unit,
) {
    val usablePlaceholder = placeholder ?: label
    var text by remember(value) { mutableStateOf(value) }
    if (enabled) {
        OutlinedTextField(
            modifier = modifier
                .fillMaxWidth(),
            value = text,
            onValueChange = { raw: String ->
                text = raw
                if (raw.isNotEmpty() && raw.isDigitsOnly()) {
                    onValueChange(text)
                }
            },
            placeholder = { Text(text = usablePlaceholder) },
            label = { Text(text = label) },
            singleLine = true,
            keyboardOptions = KeyboardOptions.Default.copy(
                keyboardType = KeyboardType.Number
            )
        )
    } else {
        Box(
            modifier = Modifier
                .fillMaxWidth()
                .background(TextFieldDefaults.colors().unfocusedContainerColor)
                .height(OutlinedTextFieldDefaults.MinHeight)
        ) {
            Row(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(16.dp, 10.dp),
                horizontalArrangement = Arrangement.SpaceBetween
            ) {
                Text(
                    color = TextFieldDefaults.colors().unfocusedLabelColor,
                    text = "$label:"
                )

                Text(
                    color = TextFieldDefaults.colors().unfocusedTextColor,
                    text = usablePlaceholder
                )
            }
        }
    }
    DefaultSpacer()
}

@Composable
fun TextAndSwitch(
    text: String,
    checked: Boolean,
    onCheckedChange: ((Boolean) -> Unit)?
) {
    ListItem(
        modifier = Modifier.fillMaxWidth(),
        headlineContent = {
            Text(
                text = text,
                maxLines = 3,
                style = MaterialTheme.typography.bodyLarge
            )
        },
        trailingContent = {
            Switch(
                checked = checked,
                onCheckedChange = onCheckedChange,
                modifier = Modifier
            )
        },
    )
}

@Composable
fun ThemeTriStateToggle(
    text: String,
    currentTheme: Theme,
    updateTheme: (Theme) -> Unit
) {
    val states = listOf(
        Theme.Auto,
        Theme.Dark,
        Theme.Light,
    )

    ListItem(
        headlineContent = {
            Text(
                text = text,
                style = MaterialTheme.typography.bodyLarge
            )
        },
        trailingContent = {
            Surface(
                shape = RoundedCornerShape(50),
                modifier = Modifier
                    .wrapContentSize()
            ) {
                Row(
                    modifier = Modifier
                        .clip(shape = RoundedCornerShape(50))
                        .background(MaterialTheme.colorScheme.surface)
                ) {
                    states.forEach { thisTheme ->
                        Text(
                            text = thisTheme.name,
                            color = MaterialTheme.colorScheme.onSurface,
                            style = MaterialTheme.typography.bodyLarge,
                            modifier = Modifier
                                .clip(shape = RoundedCornerShape(50))
                                .clickable {
                                    updateTheme(thisTheme)
                                }
                                .background(
                                    if (thisTheme == currentTheme) {
                                        MaterialTheme.colorScheme.surfaceVariant
                                    } else {
                                        MaterialTheme.colorScheme.surface
                                    }
                                )
                                .padding(
                                    vertical = 8.dp,
                                    horizontal = 16.dp,
                                ),
                        )
                    }
                }
            }
        }
    )
}

@Composable
fun TextAndTwoStateToggle(
    text: String,
    currentTheme: DistanceMeasure,
    updateTheme: (DistanceMeasure) -> Unit
) {
    val states = listOf(
        DistanceMeasure.KM,
        DistanceMeasure.MI,
    )

    ListItem(
        headlineContent = {
            Text(
                text = text,
                style = MaterialTheme.typography.bodyLarge
            )
        },
        trailingContent = {
            Surface(
                shape = RoundedCornerShape(50),
                modifier = Modifier
                    .wrapContentSize()
            ) {
                Row(
                    modifier = Modifier
                        .clip(shape = RoundedCornerShape(50))
                        .background(MaterialTheme.colorScheme.surface)
                ) {
                    states.forEach { thisTheme ->
                        Text(
                            text = thisTheme.metric,
                            color = MaterialTheme.colorScheme.onSurface,
                            style = MaterialTheme.typography.bodyLarge,
                            modifier = Modifier
                                .clip(shape = RoundedCornerShape(50))
                                .clickable {
                                    updateTheme(thisTheme)
                                }
                                .background(
                                    if (thisTheme == currentTheme) {
                                        MaterialTheme.colorScheme.surfaceVariant
                                    } else {
                                        MaterialTheme.colorScheme.surface
                                    }
                                )
                                .padding(
                                    vertical = 8.dp,
                                    horizontal = 16.dp,
                                ),
                        )
                    }
                }
            }
        }
    )
}

@Composable
fun DefaultSpacer() {
    Spacer(modifier = Modifier.size(16.dp))
}

@Composable
fun IconSpacer() {
    Spacer(modifier = Modifier.size(ButtonDefaults.IconSpacing))
}

@Composable
fun FullWidthButton(
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
    enabled: Boolean = true,
    content: @Composable () -> Unit,
) {
    Button(
        onClick = onClick,
        enabled = enabled,
        modifier = modifier.fillMaxWidth()
    ) {
        content()
    }
}

@Composable
fun FullWidthOutlinedButton(
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
    enabled: Boolean = true,
    content: @Composable () -> Unit,
) {
    OutlinedButton(
        onClick = onClick,
        enabled = enabled,
        modifier = modifier.fillMaxWidth()
    ) {
        content()
    }
}

@Composable
fun PageHeaderTextWithSpacer(text: String, helpUrlString: String?) {
    val uriHandler = LocalUriHandler.current
    Row(
        modifier = Modifier
            .fillMaxWidth()
            .padding(8.dp, 0.dp),
        horizontalArrangement = Arrangement.SpaceBetween,
        verticalAlignment = Alignment.CenterVertically
    ) {
        Text(
            text = text,
            style = MaterialTheme.typography.headlineSmall
        )
        if (!helpUrlString.isNullOrEmpty()) {
            IconButton(onClick = {
                uriHandler.openUri(helpUrlString)
            }) {
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.HelpOutline,
                    contentDescription = stringResource(R.string.icon_content_description_help)
                )
            }
        }
    }
    DefaultSpacer()
}

@Composable
fun LogEntryElement(backgroundWorkLog: BackgroundWorkLog) {
    val entryDateTime =
        backgroundWorkLog.createdInstant.toLocalDateTime(TimeZone.currentSystemDefault())
    ListItem(
        overlineContent = {
            Text(text = "${entryDateTime.year}/${entryDateTime.month.number}/${entryDateTime.day} ${entryDateTime.hour}:${entryDateTime.minute}:${entryDateTime.second}")
        },
        headlineContent = {
            Text(text = backgroundWorkLog.description)
        },
        supportingContent = {
            if (backgroundWorkLog.activityUid != null) {
                Text(text = "Activity ${backgroundWorkLog.activityUid}")
            }
        }
    )
}