package com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.SegmentedButton
import androidx.compose.material3.SegmentedButtonDefaults
import androidx.compose.material3.SingleChoiceSegmentedButtonRow
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.pointerInteropFilter
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.dp
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.components.ComponentType

@Composable
@OptIn(ExperimentalMaterial3Api::class, ExperimentalComposeUiApi::class)
fun ComponentUsageSelectorWithTextButton(
    modifier: Modifier = Modifier,
    currentBike: Bike?,
    bikes: List<Bike>,
    displayLabel: Boolean = true,
    onBikeSelected: (Bike?) -> Unit
) {
    var offset = Offset.Zero
    var bikesExpanded by remember {
        mutableStateOf(false)
    }
    Box(
        modifier = modifier
            .padding(8.dp, 0.dp)
            .wrapContentSize(Alignment.TopEnd)
            .pointerInteropFilter {
                offset = Offset(it.x, it.y)
                false
            }
    ) {
        Row(
            verticalAlignment = Alignment.CenterVertically
        ) {
            if (displayLabel) {
                Text(text = stringResource(R.string.dropdown_lbl_on_which_bike))
                DefaultSpacer()
            }
            TextButton(
                onClick = { bikesExpanded = true }
            ) {
                if (currentBike != null) {
                    Text(currentBike.name)
                } else {
                    Text(text = stringResource(R.string.dropdown_trigger_on_any_bike))
                }
            }
        }
        val density = LocalDensity.current
        DropdownMenu(
            expanded = bikesExpanded,
            offset = DpOffset(pxToDp(offset.x, density), pxToDp(offset.y, density)),
            onDismissRequest = { bikesExpanded = false }) {
            DropdownMenuItem(
                text = { Text(text = stringResource(R.string.dropdown_trigger_on_any_bike)) },
                onClick = {
                    onBikeSelected(null)
                    bikesExpanded = false
                },
                contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
            )
            bikes.forEach { bike ->
                DropdownMenuItem(
                    text = { Text(text = bike.name) },
                    onClick = {
                        onBikeSelected(bike)
                        bikesExpanded = false
                    },
                    contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
                )
            }
        }
    }
}

@Composable
fun SingleChoiceSegmentedButtonWithText(
    componentType: ComponentType,
    initCurrentBike: Bike?,
    bikes: List<Bike>,
    numberOfComponentsOnShelf: Int,
    numberOfRetiredComponents: Int,
    modifier: Modifier = Modifier,
    onComponentTypeSwitched: (ComponentType) -> Unit,
    onSelected: (Bike?) -> Unit,
    onNavigateToAnalysis: () -> Unit
) {
    var currentBike = initCurrentBike

    SingleChoiceSegmentedButtonRow(
        modifier = modifier.fillMaxWidth()
    ) {
        SegmentedButton(
            shape = SegmentedButtonDefaults.itemShape(
                index = 0,
                count = 3
            ),
            onClick = { onComponentTypeSwitched(ComponentType.ON_SHELF) },
            selected = (componentType == ComponentType.ON_SHELF),
            enabled = numberOfComponentsOnShelf > 0,
            label = { Text(stringResource(R.string.dropdown_trigger_shelf)) }
        )
        SegmentedButton(
            shape = SegmentedButtonDefaults.itemShape(
                index = 1,
                count = 3
            ),
            onClick = { onComponentTypeSwitched(ComponentType.ON_BIKE) },
            selected = (componentType == ComponentType.ON_BIKE),
            label = { Text(text = stringResource(R.string.dropdown_trigger_on_bike)) }
        )
        SegmentedButton(
            shape = SegmentedButtonDefaults.itemShape(
                index = 2,
                count = 3
            ),
            onClick = { onComponentTypeSwitched(ComponentType.RETIRED) },
            selected = (componentType == ComponentType.RETIRED),
            enabled = numberOfRetiredComponents > 0,
            label = { Text(text = stringResource(R.string.dropdown_trigger_retired)) }
        )
    }
    AnimatedVisibility(
        visible = componentType == ComponentType.ON_BIKE
    ) {
        DefaultSpacer()
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.End
        ) {
            ComponentUsageSelectorWithTextButton(
                currentBike = currentBike,
                bikes = bikes,
            ) { bike ->
                if (bike != currentBike) {
                    currentBike = bike
                    onSelected(bike)
                }
            }
        }
    }
    AnimatedVisibility(
        visible = componentType == ComponentType.RETIRED && numberOfRetiredComponents > 0
    ) {
        FullWidthOutlinedButton(
            modifier = Modifier.padding(0.dp, 16.dp, 0.dp, 0.dp),
            onClick = {
                onNavigateToAnalysis()
            }
        ) {
            Text(text = stringResource(R.string.menu_item_analyse_components))
        }
    }
}
