package com.exner.tools.jkbikemechanicaldisasterprevention.ui.customComponents

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.material3.Button
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuAnchorType
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.input.pointer.pointerInteropFilter
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.dp
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DefaultBikeSelectorWithSpacer(
    value: String,
    label: String,
    onMenuItemClick: (Long?) -> Unit,
    bikes: List<Bike>
) {
    var bikeSelectorExpanded by remember { mutableStateOf(false) }
    ExposedDropdownMenuBox(
        expanded = bikeSelectorExpanded,
        onExpandedChange = { bikeSelectorExpanded = !bikeSelectorExpanded }
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth(),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween
        ) {
            OutlinedTextField(
                // The `menuAnchor` modifier must be passed to the text field for correctness.
                modifier = Modifier
                    .menuAnchor(
                        type = ExposedDropdownMenuAnchorType.PrimaryEditable,
                        enabled = true
                    )
                    .fillMaxWidth()
                    .padding(8.dp),
                readOnly = true,
                value = value,
                onValueChange = {},
                label = { Text(text = label) },
                trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = bikeSelectorExpanded) },
            )
        }
        ExposedDropdownMenu(
            expanded = bikeSelectorExpanded,
            onDismissRequest = { bikeSelectorExpanded = false }) {
            DropdownMenuItem(
                text = { Text(text = stringResource(R.string.dropdown_item_none)) },
                onClick = {
                    onMenuItemClick(null)
                    bikeSelectorExpanded = false
                },
                contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
            )
            bikes.forEach { bike ->
                DropdownMenuItem(
                    text = { Text(text = bike.name) },
                    onClick = {
                        onMenuItemClick(bike.uid)
                        bikeSelectorExpanded = false
                    },
                    contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
                )
            }
        }
    }
    DefaultSpacer()
}

@Composable
@OptIn(ExperimentalMaterial3Api::class, ExperimentalComposeUiApi::class)
fun BikeSelector(
    modifier: Modifier = Modifier,
    currentBike: Bike?,
    bikes: List<Bike>,
    label: String?,
    onBikeSelected: (Bike?) -> Unit
) {
    var offset = Offset.Zero
    var bikesExpanded by remember { mutableStateOf(false) }
    Box(
        modifier = modifier
            .padding(8.dp, 0.dp, 8.dp, 8.dp)
            .wrapContentSize(Alignment.TopEnd)
            .pointerInteropFilter {
                offset = Offset(it.x, it.y)
                false
            }
    ) {
        FlowRow(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            itemVerticalAlignment = Alignment.CenterVertically,
            maxItemsInEachRow = 2
        ) {
            if (label != null) {
                Text(
                    text = label,
                    maxLines = 2
                )
            }
            Spacer(modifier = Modifier.size(24.dp))
            Spacer(modifier = Modifier.size(24.dp))
            Button(
                onClick = { bikesExpanded = true },
            ) {
                if (currentBike != null) {
                    Text(text = currentBike.name)
                } else {
                    Text(
                        text = stringResource(R.string.dropdown_trigger_select_a_bike),
                    )
                }
            }
        }
        val density = LocalDensity.current
        DropdownMenu(
            expanded = bikesExpanded,
            offset = DpOffset(pxToDp(offset.x, density), pxToDp(offset.y, density)),
            onDismissRequest = { bikesExpanded = false }) {
            DropdownMenuItem(
                text = { Text(text = stringResource(R.string.dropdown_item_all_bikes)) },
                onClick = {
                    onBikeSelected(null)
                    bikesExpanded = false
                },
                contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
            )
            bikes.forEach { bike ->
                DropdownMenuItem(
                    text = { Text(text = bike.name) },
                    onClick = {
                        onBikeSelected(bike)
                        bikesExpanded = false
                    },
                    contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
                )
            }
        }
    }
}

@Composable
@OptIn(ExperimentalMaterial3Api::class, ExperimentalComposeUiApi::class, ExperimentalTime::class)
fun BikeSelectorWithTextButtonAndAddBikeOption(
    modifier: Modifier = Modifier,
    currentBike: Bike?,
    bikes: List<Bike>,
    displayLabel: Boolean = true,
    addFromIntegrationLabelString: String,
    onBikeSelected: (Bike?) -> Unit
) {
    var offset = Offset.Zero
    var bikesExpanded by remember {
        mutableStateOf(false)
    }

    val context = LocalContext.current

    Box(
        modifier = modifier
            .padding(8.dp, 0.dp)
            .wrapContentSize(Alignment.TopEnd)
            .pointerInteropFilter {
                offset = Offset(it.x, it.y)
                false
            }
    ) {
        Row(
            verticalAlignment = Alignment.CenterVertically
        ) {
            if (displayLabel) {
                Text(text = stringResource(R.string.dropdown_lbl_which_bike))
                DefaultSpacer()
            }
            TextButton(
                onClick = { bikesExpanded = true }
            ) {
                if (currentBike != null) {
                    Text(text = currentBike.name)
                } else {
                    Text(text = stringResource(R.string.dropdown_trigger_select_a_bike))
                }
            }
        }
        val density = LocalDensity.current
        DropdownMenu(
            expanded = bikesExpanded,
            offset = DpOffset(pxToDp(offset.x, density), pxToDp(offset.y, density)),
            onDismissRequest = { bikesExpanded = false }) {
            DropdownMenuItem(
                text = { Text(text = stringResource(R.string.dropdown_item_all_bikes)) },
                onClick = {
                    onBikeSelected(null)
                    bikesExpanded = false
                },
                contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
            )
            bikes.forEach { bike ->
                DropdownMenuItem(
                    text = { Text(text = bike.name) },
                    onClick = {
                        onBikeSelected(bike)
                        bikesExpanded = false
                    },
                    contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding,
                )
            }
            DropdownMenuItem(
                text = { Text(text = addFromIntegrationLabelString) },
                onClick = {
                    onBikeSelected(
                        Bike(
                            name = addFromIntegrationLabelString,
                            buildDate = Clock.System.now()
                                .toLocalDateTime(TimeZone.currentSystemDefault()).date,
                            mileage = 0,
                            lastUsedDate = null,
                            isElectric = false,
                            isActive = true,
                            retirementDate = null,
                            stravaGearId = null,
                            intervalsGearId = null,
                            uid = 0L
                        )
                    )
                    bikesExpanded = false
                },
                contentPadding = ExposedDropdownMenuDefaults.ItemContentPadding
            )
        }
    }
}

fun pxToDp(pixels: Float, density: Density): Dp {
    val dp = with(density) { pixels.toDp() }
    return dp
}
