package com.exner.tools.jkbikemechanicaldisasterprevention.ui

import android.content.ContentUris
import android.content.Context
import android.net.Uri
import android.provider.MediaStore
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.tools.Export
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.stateIn
import javax.inject.Inject

private const val TAG = "MainVM"

@HiltViewModel
class MainViewModel @Inject constructor(
    val repository: KJsRepository,
    userPreferencesManager: UserPreferencesManager
) : ViewModel() {

    val userSelectedTheme: StateFlow<Theme> = userPreferencesManager.theme().stateIn(
        scope = viewModelScope,
        started = SharingStarted.WhileSubscribed(stopTimeoutMillis = 5_000),
        initialValue = Theme.Auto
    )

    val stravaEnabled: StateFlow<Boolean> = userPreferencesManager.stravaEnabled().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    val intervalsEnabled: StateFlow<Boolean> = userPreferencesManager.intervalsEnabled().stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(),
        false
    )

    fun createBackup(
        context: Context,
        successCallback: (() -> Unit)? = null,
        failureCallback: ((String) -> Unit)? = null
    ) {
        val export = Export(
            repository = repository,
            context = context
        )
        export.commitExport(
            exportType = "backup",
            exportBuiltInTemplates = false,
            exportCompletedActivitiesAndRides = true,
            successCallback = successCallback,
            failureCallback = failureCallback
        )
    }

    fun pruneBackups(
        context: Context,
        successCallback: (() -> Unit)? = null,
        failureCallback: ((String) -> Unit)? = null
    ) {
        Log.d(TAG, "Initiating pruning of automatic backup files...")
        val projection = arrayOf(
            MediaStore.Downloads._ID,
            MediaStore.Downloads.DISPLAY_NAME
        )
        val selection = "${MediaStore.Downloads.DISPLAY_NAME} LIKE ?" // ${MediaStore.Downloads.CONTENT_TYPE} = ${MediaStore.}
        val selectionArgs = arrayOf(
            "jkbike-backup-%"
        )
        val sortOrder = "${MediaStore.Downloads.DISPLAY_NAME} ASC"

        val resolver = context.contentResolver
        val query = resolver.query(
            MediaStore.Downloads.EXTERNAL_CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            sortOrder
        )
        if (query != null) {
            Log.d(TAG, "Found ${query.count} backups.")
            if (query.count > 10) {
                Log.d(TAG, "Pruning backups...")
                var numBackups = query.count
                query.use { cursor ->
                    val idColumn = cursor.getColumnIndexOrThrow(MediaStore.Downloads._ID)
                    val nameColumn =
                        cursor.getColumnIndexOrThrow(MediaStore.Downloads.DISPLAY_NAME)
                    while (cursor.moveToNext()) {
                        val id = cursor.getLong(idColumn)
                        val name = cursor.getString(nameColumn)
                        val contentUri: Uri = ContentUris.withAppendedId(
                            MediaStore.Downloads.EXTERNAL_CONTENT_URI,
                            id
                        )
                        // delete it?
                        if (numBackups > 10) {
                            Log.d(TAG, "Deleting $name...")
                            resolver.delete(contentUri, null, null)
                            numBackups--
                        }
                    }
                }
            }
        }
    }
}