package com.exner.tools.jkbikemechanicaldisasterprevention.ui

import android.content.Context
import androidx.activity.ComponentActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Link
import androidx.compose.material.icons.filled.LinkOff
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.windowsizeclass.WindowSizeClass
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.helpers.NavigationStyle
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithIntervalsViewModel
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.integrations.SyncWithStravaViewModel
import com.ramcosta.composedestinations.DestinationsNavHost
import com.ramcosta.composedestinations.generated.NavGraphs
import com.ramcosta.composedestinations.generated.destinations.ChecklistDestination
import com.ramcosta.composedestinations.generated.destinations.HomeDestination
import com.ramcosta.composedestinations.navigation.DestinationsNavigator
import com.ramcosta.composedestinations.navigation.dependency
import com.ramcosta.composedestinations.rememberNavHostEngine
import com.ramcosta.composedestinations.spec.DestinationSpec
import com.ramcosta.composedestinations.utils.currentDestinationAsState
import com.ramcosta.composedestinations.utils.rememberDestinationsNavigator

@Composable
fun KJsGlobalScaffold(
    activity: ComponentActivity,
    windowSizeClass: WindowSizeClass,
    kJsGlobalScaffoldViewModel: KJsGlobalScaffoldViewModel = hiltViewModel(),
    syncWithStravaViewModel: SyncWithStravaViewModel = hiltViewModel(),
    syncWithIntervalsViewModel: SyncWithIntervalsViewModel = hiltViewModel(),
) {
    val engine = rememberNavHostEngine()
    val navController = engine.rememberNavController()
    val destinationsNavigator = navController.rememberDestinationsNavigator()
    val destination = navController.currentDestinationAsState().value

    val navigationStyle =
        NavigationStyle.getNavigationStyleForWidthSizeClass(windowSizeClass.widthSizeClass)

    val destinationTitle by kJsGlobalScaffoldViewModel.destinationTitle.collectAsStateWithLifecycle()

    val stravaEnabled by kJsGlobalScaffoldViewModel.stravaEnabled.collectAsStateWithLifecycle()
    val isStravaConnected by syncWithStravaViewModel.isStravaConnected.collectAsStateWithLifecycle()

    val intervalsEnabled by kJsGlobalScaffoldViewModel.intervalsEnabled.collectAsStateWithLifecycle()
    val isIntervalsConnected by syncWithIntervalsViewModel.isIntervalsConnected.collectAsStateWithLifecycle()

    Scaffold(
        topBar = {
            KJsTopBar(
                destination,
                destinationsNavigator,
                navigationStyle,
                destinationTitle,
                stravaEnabled || intervalsEnabled,
                isStravaConnected || isIntervalsConnected,
                connectIntegration = { context ->
                    if (stravaEnabled) {
                        syncWithStravaViewModel.connect(context)
                    }
                    if (intervalsEnabled) {
                        syncWithIntervalsViewModel.connect(context)
                    }
                },
                disconnectIntegration = { context ->
                    if (stravaEnabled) {
                        syncWithStravaViewModel.disconnectStrava()
                    }
                    if (intervalsEnabled) {
                        syncWithIntervalsViewModel.disconnect()
                    }
                }
            )
        }
    ) { innerPadding ->
        val newPadding = PaddingValues.Absolute(
            innerPadding.calculateLeftPadding(LayoutDirection.Ltr),
            innerPadding.calculateTopPadding(),
            innerPadding.calculateRightPadding(LayoutDirection.Ltr),
            0.dp
        )
        Surface(
            modifier = Modifier
                .padding(newPadding)
                .imePadding()
                .consumeWindowInsets(newPadding)
        ) {
            DestinationsNavHost(
                navController = navController,
                navGraph = NavGraphs.root,
                dependenciesContainerBuilder = { //this: DependenciesContainerBuilder<*>
                    dependency(windowSizeClass)
                    // make this ViewModel available to all destinations, and scoped on the Activity
                    dependency(hiltViewModel<KJsGlobalScaffoldViewModel>(activity))
                },
            ) {
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun KJsTopBar(
    destination: DestinationSpec?,
    destinationsNavigator: DestinationsNavigator,
    navigationStyle: NavigationStyle,
    destinationTitle: String,
    isIntegrationEnabled: Boolean = false,
    isIntegrationConnected: Boolean = false,
    connectIntegration: (Context) -> Unit = {},
    disconnectIntegration: (Context) -> Unit = {}
) {

    when (navigationStyle) {
        NavigationStyle.BOTTOM_BAR -> { // compact
            TopAppBar(
                title = { Text(text = stringResource(R.string.top_bar_title)) },
                navigationIcon = {
                    KJsGlobalBackNavigationIcon(destination, destinationsNavigator)
                },
                actions = {
                    if (isIntegrationEnabled) {
                        val context = LocalContext.current
                        if (isIntegrationConnected) {
                            Icon(
                                imageVector = Icons.Default.Link,
                                contentDescription = stringResource(R.string.btn_desc_connected),
                                tint = Color.Green,
                                modifier = Modifier.clickable(
                                    enabled = true, // TODO
                                    onClick = {
                                        disconnectIntegration(context)
                                    }
                                )
                            )
                        } else {
                            Icon(
                                imageVector = Icons.Default.LinkOff,
                                contentDescription = stringResource(R.string.btn_desc_disconnected),
                                tint = Color.Red,
                                modifier = Modifier.clickable(
                                    enabled = true,
                                    onClick = {
                                        connectIntegration(context)
                                    }
                                )
                            )
                        }
                    }
                    KJsGlobalMainMenuAction(destination, destinationsNavigator)
                }
            )
        }

        NavigationStyle.LEFT_RAIL, NavigationStyle.LEFT_DRAWER -> { // medium
            val title =
                if (navigationStyle == NavigationStyle.LEFT_RAIL && destinationTitle.isNotBlank()) {
                    stringResource(R.string.top_bar_title) + " - " + destinationTitle
                } else {
                    stringResource(R.string.top_bar_title_long) + if (navigationStyle == NavigationStyle.LEFT_DRAWER && destinationTitle.isNotBlank()) {
                        " - $destinationTitle"
                    } else {
                        ""
                    }
                }
            CenterAlignedTopAppBar(
                title = {
                    Text(text = title)
                },
                navigationIcon = {
                    KJsGlobalBackNavigationIcon(destination, destinationsNavigator)
                },
                actions = {
                    if (isIntegrationEnabled) {
                        val context = LocalContext.current
                        if (isIntegrationConnected) {
                            Icon(
                                imageVector = Icons.Default.Link,
                                contentDescription = stringResource(R.string.btn_desc_connected),
                                tint = Color.Green,
                                modifier = Modifier.clickable(
                                    enabled = true, // TODO
                                    onClick = {
                                        disconnectIntegration(context)
                                    }
                                )
                            )
                        } else {
                            Icon(
                                imageVector = Icons.Default.LinkOff,
                                contentDescription = stringResource(R.string.btn_desc_disconnected),
                                tint = Color.Red,
                                modifier = Modifier.clickable(
                                    enabled = true,
                                    onClick = {
                                        connectIntegration(context)
                                    }
                                )
                            )
                        }
                    }
                    KJsGlobalMainMenuAction(destination, destinationsNavigator)
                },
            )
        }
    }
}

@Composable
private fun KJsGlobalBackNavigationIcon(
    destination: DestinationSpec?,
    destinationsNavigator: DestinationsNavigator
) {
    when (destination) {
        HomeDestination -> {
            // no back button here
        }

        ChecklistDestination -> {
            IconButton(onClick = { destinationsNavigator.popBackStack(HomeDestination, inclusive = false) }) {
                Icon(
                    imageVector = Icons.AutoMirrored.Default.ArrowBack,
                    contentDescription = stringResource(R.string.btn_desc_back)
                )
            }
        }

        else -> {
            IconButton(onClick = { destinationsNavigator.navigateUp() }) {
                Icon(
                    imageVector = Icons.AutoMirrored.Default.ArrowBack,
                    contentDescription = stringResource(R.string.btn_desc_back)
                )
            }
        }
    }
}
