package com.exner.tools.jkbikemechanicaldisasterprevention.scheduler

import android.content.Context
import android.util.Log
import android.util.Log.w
import androidx.hilt.work.HiltWorker
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.NetworkType
import androidx.work.PeriodicWorkRequest
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkerParameters
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.BackgroundWorkLog
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Bike
import com.exner.tools.jkbikemechanicaldisasterprevention.network.intervals.IntervalsApiManager
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import java.util.concurrent.TimeUnit
import kotlin.math.roundToInt
import kotlin.time.Clock

private const val TAG = "UBDFIWorker"

@HiltWorker
class UpdateBikeDistancesFromIntegrationWorker @AssistedInject constructor(
    @Assisted private val appContext: Context,
    @Assisted workerParams: WorkerParameters,
    val repository: KJsRepository,
    val userPreferencesManager: UserPreferencesManager,
    val intervalsApiManager: IntervalsApiManager
) : CoroutineWorker(appContext, workerParams) {

    private val clientId = "98"

    private val metricsFactor = 1.609344f

    override suspend fun doWork(): Result {
        Log.d(TAG, "Starting periodic bike distances check...")

        // Do the work here
        val isStravaEnabled = userPreferencesManager.stravaEnabled().firstOrNull()
        val isIntervalsEnabled = userPreferencesManager.intervalsEnabled().firstOrNull()
        val isUpdateDistances = userPreferencesManager.updateBikeDistancesPeriodically().firstOrNull()

        // more data
        val metricsPreferenceLocal = userPreferencesManager.distanceMeasure().firstOrNull()
        val metricsPreferenceInterval =
            userPreferencesManager.distanceMeasureIntervals().firstOrNull()

        if (isUpdateDistances == true) {
            if (isStravaEnabled == true) {
//                if (syncWithStravaViewModel.isStravaConnected.value == true) {
//                    syncWithStravaViewModel.updateDistancesFromStrava(appContext, {
//                        logSuccess()
//                    }, {
//                        logError()
//                    })
//                }
            }
            if (isIntervalsEnabled == true) {
                if (intervalsApiManager.isConnected() == true) {
                    CoroutineScope(Dispatchers.IO).launch {
                        val bikes = repository.getAllBikes()
                        updateMileageOnBikeAndComponents(
                            context = appContext,
                            bikes = bikes,
                            metricsPreferenceLocal = metricsPreferenceLocal,
                            metricsPreferenceInterval = metricsPreferenceInterval,
                            onSuccess = {
                                logSuccess()
                            },
                            onError = { code ->
                                logError(code)
                            }
                        )
                    }
                }
            }
        } else {
            Log.d(TAG, "Settings says: do not update periodically, so not doing it.")
        }

        // Indicate whether the work finished successfully with the Result
        return Result.success()
    }

    fun logSuccess() {
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Log creation of update job...")
            val logEntry = BackgroundWorkLog(
                createdInstant = Clock.System.now(),
                source = "ubdfi", // see JKWorkers class
                createdComponentUid = null,
                activityUid = null,
                description = "Started update of distances for all bikes"
            )
            repository.insertBackgroundWorkLog(logEntry)
        }
    }

    fun logError(code: Int?) {
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Log creation of activity...")
            val logEntry = BackgroundWorkLog(
                createdInstant = Clock.System.now(),
                source = "ubdfi", // see JKWorkers class
                createdComponentUid = null,
                activityUid = null,
                description = "Failed to start update of distances for all bikes - $code"
            )
            repository.insertBackgroundWorkLog(logEntry)
        }
    }

    suspend private fun updateMileageOnBikeAndComponents(
        context: Context,
        bikes: List<Bike>,
        metricsPreferenceLocal: DistanceMeasure?,
        metricsPreferenceInterval: DistanceMeasure?,
        onSuccess: () -> Unit,
        onError: (Int) -> Unit
    ) {
        if (bikes.isNotEmpty()) {
            intervalsApiManager.retrieveGear(
                context = context,
                onResult = { gearList ->
                    gearList.forEach { gear ->
                        var fixedGear = gear
                        // new gear may have a different metric! Fix that rirst
                        Log.d(
                            "SWIVM",
                            "Conversion? $metricsPreferenceLocal/$metricsPreferenceInterval ${fixedGear.distance}"
                        )
                        if (metricsPreferenceLocal != metricsPreferenceInterval) {
                            // yup, needs fixing
                            val distanceFromGear = fixedGear.distance
                            if (metricsPreferenceLocal == DistanceMeasure.KM && metricsPreferenceInterval == DistanceMeasure.MI) {
                                // mi -> km, multiply
                                fixedGear = fixedGear.copy(
                                    distance = distanceFromGear?.times(metricsFactor)
                                )
                            } else if (metricsPreferenceLocal == DistanceMeasure.MI && metricsPreferenceInterval == DistanceMeasure.KM) {
                                // km -> mi, divide
                                fixedGear = fixedGear.copy(
                                    distance = distanceFromGear?.div(metricsFactor)
                                )
                            } else {
                                // Well... this should NEVER happen!
                                // but I do not want to crash the app, either...
                                // people will report when metrics are off
                            }
                        }
                        // now use it
                        CoroutineScope(Dispatchers.IO).launch {
                            if (fixedGear.id != null) {
                                val bike = repository.getBikeForIntervalsGearId(fixedGear.id)
                                if (bike != null) {
                                    val beforeDistance = bike.mileage
                                    val newDistance = fixedGear.distance
                                    if (newDistance != null) {
                                        val newRealDistance =
                                            (fixedGear.distance / 1000).roundToInt()
                                        if (beforeDistance < newRealDistance) {
                                            val difference = newRealDistance - beforeDistance
                                            Log.d(TAG, "Bike ${bike.name} has $difference additional distance. Updating bike and components (if any)...")
                                            // add this distance to every component attached to this bike
                                            val allComponentsForBike =
                                                repository.getAllComponentsForBike(bike.uid)
                                            allComponentsForBike.forEach { component ->
                                                val newMileage =
                                                    difference + (component.currentMileage ?: 0)
                                                val updatedComponent = component.copy(
                                                    currentMileage = newMileage
                                                )
                                                repository.updateComponent(updatedComponent)
                                            }
                                        }
                                        // update the bike itself
                                        val updatedBike = bike.copy(
                                            mileage = newRealDistance
                                        )
                                        repository.updateBike(updatedBike)
                                    }
                                }
                            }
                        }
                    }
                    // success
                    Log.d(TAG, "Successfully retrieved distances and started update of data locally.")
                    onSuccess()
                },
                onError = { code ->
                    w(TAG, "Error retrieving gear $code")
                    onError(code)
                }
            )
        } else {
            w(TAG, "There are no bikes to retrieve")
            onError(404)
        }
    }

    companion object {
        fun createWorkRequest(): PeriodicWorkRequest {
            val constraintsDistanceUpdate = Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .setRequiresStorageNotLow(true)
                .setRequiredNetworkType(NetworkType.NOT_ROAMING)
                .build()
            val distanceUpdaterWorkRequest: PeriodicWorkRequest =
                PeriodicWorkRequestBuilder<UpdateBikeDistancesFromIntegrationWorker>(
                    repeatInterval = 1,
                    repeatIntervalTimeUnit = TimeUnit.DAYS,
//                flexTimeInterval = 4,
//                flexTimeIntervalUnit = TimeUnit.HOURS
                )
                    .setConstraints(constraintsDistanceUpdate)
                    .setInitialDelay(15, TimeUnit.SECONDS)
                    .build()
            return distanceUpdaterWorkRequest
        }
    }
}