package com.exner.tools.jkbikemechanicaldisasterprevention.scheduler

import android.content.Context
import android.util.Log
import androidx.hilt.work.HiltWorker
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.PeriodicWorkRequest
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkerParameters
import com.exner.tools.jkbikemechanicaldisasterprevention.R
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.Activity
import com.exner.tools.jkbikemechanicaldisasterprevention.database.entities.BackgroundWorkLog
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.notifications.Notifications
import dagger.assisted.Assisted
import dagger.assisted.AssistedInject
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.datetime.DateTimeUnit
import kotlinx.datetime.TimeZone
import kotlinx.datetime.plus
import kotlinx.datetime.todayIn
import java.util.concurrent.TimeUnit
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

private const val TAG = "CCIACAWorker"

@HiltWorker
class CheckComponentsIntervalsAndCreateActivitiesWorker @AssistedInject constructor(
    @Assisted private val appContext: Context,
    @Assisted workerParams: WorkerParameters,
    val repository: KJsRepository,
    val userPreferencesManager: UserPreferencesManager
) : CoroutineWorker(appContext, workerParams) {

    override suspend fun doWork(): Result {
        val isUpdateDistances =
            userPreferencesManager.updateBikeDistancesPeriodically().firstOrNull()

        if (isUpdateDistances == true) {
            Log.d(TAG, "Starting periodic components check...")

            // Do the work here
            checkAndCreate(appContext, repository)
        } else {
            Log.d(TAG, "Settings say: do not create activities, so not doing it.")
        }

        // Indicate whether the work finished successfully with the Result
        return Result.success()
    }

    /**
     * When an activity is generated from here, it will receive the componentUid
     *
     * When such an activity is checked / ticked off
     * - component lastCheckDate is updated
     * - component currentMileage is updated from bike
     * - component wear is updated (UI needed!)
     * - is taken off bike and retireDate updated when wear is "dead"
     */

    @OptIn(ExperimentalTime::class)
    suspend private fun checkAndCreate(
        appContext: Context,
        repository: KJsRepository
    ) {
        val isEnableNotifications = userPreferencesManager.enableNotifications().firstOrNull()

        val allComponents = repository.getAllComponents()
        Log.d(TAG, "Found ${allComponents.size} components.")
        val today = Clock.System.todayIn(TimeZone.currentSystemDefault())
        allComponents.forEach { component ->
            Log.d(TAG, "Component ${component.name}")
            if (component.bikeUid != null) { // is attached to a bike
                Log.d(TAG, "  is attached to bike ${component.bikeUid}")
                if (0 > (component.firstUseDate?.compareTo(today) ?: 0)) {
                    Log.d(TAG, "  is being or has been used.")
                    // first_use_date is in the past! This component is being used!
                    var isItTime = false
                    var isItTheMileage = false
                    var isItTheTime = false
                    // check mileage
                    if (component.checkIntervalMiles != null) {
                        component.currentMileage?.let { currentMileage ->
                            if (currentMileage > (component.checkIntervalMiles + (component.lastCheckMileage
                                    ?: 0))
                            ) {
                                if (component.lastCheckNotificationMileage == null || currentMileage > component.lastCheckNotificationMileage) {
                                    Log.d(TAG, "  mileage test positive.")
                                    isItTime = true
                                    isItTheMileage = true
                                } else {
                                    Log.d(TAG, "  mileage test positive, but we notified already.")
                                }
                            }
                        }
                    }
                    // check time
                    if (component.checkIntervalDays != null && component.lastCheckDate != null) {
                        if (component.lastCheckDate.plus(
                                component.checkIntervalDays,
                                DateTimeUnit.DAY
                            ) < today
                        ) {
                            if (component.lastCheckNotificationDate == null || today > component.lastCheckNotificationDate) {
                                Log.d(TAG, "  date test positive.")
                                isItTime = true
                                isItTheTime = true
                            } else {
                                Log.d(TAG, "  date test positive, but we notified already today.")
                            }
                        }
                    }
                    // if either of them were due, do it
                    if (isItTime) {
                        // record that we (would have) notified
                        val updatedComponent = component.copy(
                            lastCheckNotificationMileage = component.currentMileage,
                            lastCheckNotificationDate = today
                        )
                        Log.d(TAG, "Saving mileage/date to component")
                        repository.updateComponent(updatedComponent)
                        val bike = repository.getBikeByUid(component.bikeUid)
                        // create activity
                        val title = component.titleForAutomaticActivities
                            ?: (appContext.getString(R.string.check_component) + " " + component.name)
                        val activity = Activity(
                            title = title,
                            description = component.description,
                            isCompleted = false,
                            bikeUid = component.bikeUid,
                            isEBikeSpecific = bike?.isElectric == true,
                            rideLevel = null,
                            rideUid = null,
                            createdInstant = Clock.System.now(),
                            dueDate = today.plus(3, DateTimeUnit.DAY),
                            doneInstant = null,
                            componentUid = component.uid,
                            uid = 0L,
                        )
                        Log.d(TAG, "Creating activity $activity for component $component...")
                        repository.insertActivity(activity)

                        Log.d(TAG, "Log creation of activity...")
                        val logEntry = BackgroundWorkLog(
                            createdInstant = Clock.System.now(),
                            source = "cciaca", // see JKWorkers class
                            createdComponentUid = component.uid,
                            activityUid = activity.uid,
                            description = "Activity ${activity.uid}/${activity.title} created for component ${component.uid}/${component.name}"
                        )
                        repository.insertBackgroundWorkLog(logEntry)

                        if (isEnableNotifications == true && component.notifyWhenActivityCreated == true) {
                            // TODO create notification
                            val notifications = Notifications()
                            notifications.makeNotification(
                                context = appContext,
                                title = component.titleForAutomaticActivities ?: "Activity created",
                                content = component.name + " - " + if (isItTheMileage) {
                                    "distance reached: ${component.checkIntervalMiles}"
                                } else if (isItTheTime) {
                                    "time reached: reached ${component.checkIntervalDays}"
                                } else {
                                    "should be checked"
                                },
                                notificationId = component.uid.toInt()
                            )
                        }
                    }
                }
            }
        }
    }

    companion object {
        fun createWorkRequest(): PeriodicWorkRequest {
            val constraints = Constraints.Builder()
                .setRequiresBatteryNotLow(true)
                .setRequiresStorageNotLow(true)
                .build()
            val schedulerWorkRequest: PeriodicWorkRequest =
                PeriodicWorkRequestBuilder<CheckComponentsIntervalsAndCreateActivitiesWorker>(
                    repeatInterval = 1,
                    repeatIntervalTimeUnit = TimeUnit.DAYS,
//                flexTimeInterval = 13,
//                flexTimeIntervalUnit = TimeUnit.HOURS
                )
                    .setConstraints(constraints)
                    .setInitialDelay(30, TimeUnit.SECONDS)
                    .build()
            return schedulerWorkRequest
        }
    }
}
