package com.exner.tools.jkbikemechanicaldisasterprevention.preferences

import android.content.Context
import android.util.Log
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.emptyPreferences
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.Theme
import com.exner.tools.jkbikemechanicaldisasterprevention.ui.jkbike.DistanceMeasure
import dagger.hilt.android.qualifiers.ApplicationContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map
import javax.inject.Inject
import javax.inject.Singleton

private val Context.dataStore by preferencesDataStore("preferences")

@Singleton
class UserPreferencesManager @Inject constructor(
    @ApplicationContext appContext: Context
) {
    private val userDataStorePreferences = appContext.dataStore

    fun theme(): Flow<Theme> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            Theme.valueOf(preferences[KEY_THEME] ?: Theme.Auto.name)
        }
    }

    suspend fun setTheme(newTheme: Theme) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_THEME] = newTheme.name
        }
    }

    fun distanceMeasure(): Flow<DistanceMeasure> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            DistanceMeasure.valueOf(preferences[KEY_SI_OR_WEIRDO] ?: DistanceMeasure.KM.toString())
        }
    }

    suspend fun setDistanceMeasure(newDistanceMeasure: DistanceMeasure) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_SI_OR_WEIRDO] = newDistanceMeasure.toString()
        }
    }

    fun stravaEnabled(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_STRAVA_ENABLED] ?: false
        }
    }

    suspend fun setStravaEnabled(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_STRAVA_ENABLED] = enabled
        }
    }

    fun stravaAuthStateAsString(): Flow<String?> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_STRAVA_AUTH_STATE]
        }
    }

    suspend fun setStravaAuthStateAsString(authStateAsString: String?) {
        if (!authStateAsString.isNullOrEmpty()) {
            userDataStorePreferences.edit { preferences ->
                preferences[KEY_STRAVA_AUTH_STATE] = authStateAsString
            }
        }
    }

    fun advancedFunctionality(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_ADVANCED_FUNCTIONALITY] ?: false
        }
    }

    suspend fun setAdvancedFunctionality(advanced: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_ADVANCED_FUNCTIONALITY] = advanced
        }
    }

    fun stravaAllowedAllProfileData(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_STRAVA_ALLOWED_ALL_PROFILE_DATA] ?: false
        }
    }

    suspend fun setStravaAllowedAllProfileData(allowed: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_STRAVA_ALLOWED_ALL_PROFILE_DATA] = allowed
        }
    }

    fun intervalsEnabled(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_INTERVALS_ENABLED] ?: false
        }
    }

    suspend fun setIntervalsEnabled(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_INTERVALS_ENABLED] = enabled
        }
    }

    fun intervalsAuthStateAsString(): Flow<String?> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_INTERVALS_AUTH_STATE]
        }
    }

    suspend fun setIntervalsAuthStateAsString(authStateAsString: String?) {
        if (!authStateAsString.isNullOrEmpty()) {
            userDataStorePreferences.edit { preferences ->
                preferences[KEY_INTERVALS_AUTH_STATE] = authStateAsString
            }
        }
    }

    fun intervalsAllowedAthleteSettings(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_INTERVALS_ALLOWED_READ_ATHLETE_SETTINGS] ?: false
        }
    }

    suspend fun setIntervalsAllowedAthleteSettings(allowed: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_INTERVALS_ALLOWED_READ_ATHLETE_SETTINGS] = allowed
        }
    }

    fun distanceMeasureIntervals(): Flow<DistanceMeasure> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            DistanceMeasure.valueOf(
                preferences[KEY_INTERVALS_SI_OR_WEIRDO] ?: DistanceMeasure.KM.toString()
            )
        }
    }

    suspend fun setDistanceMeasureIntervals(newDistanceMeasure: DistanceMeasure) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_INTERVALS_SI_OR_WEIRDO] = newDistanceMeasure.toString()
        }
    }

    fun templateActivityLanguage(): Flow<String> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_TEMPLATE_ACTIVITY_LANGUAGE] ?: "default" // meaning use system language
        }
    }

    suspend fun setTemplateActivityLanguage(newLanguage: String) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_TEMPLATE_ACTIVITY_LANGUAGE] = newLanguage
        }
    }

    fun automaticallyUpdateTemplatesOnAppStart(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_AUTOMATICALLY_UPDATE_TEMPLATES_ON_APP_START] ?: true
        }
    }

    suspend fun setAutomaticallyUpdateTemplateActivitiesOnAppStart(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_AUTOMATICALLY_UPDATE_TEMPLATES_ON_APP_START] = enabled
        }
    }

    fun updateBikeDistancesPeriodically(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_UPDATE_BIKE_DISTANCES_PERIODICALLY] ?: false
        }
    }

    suspend fun setUpdateBikeDistancesPeriodically(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_UPDATE_BIKE_DISTANCES_PERIODICALLY] = enabled
        }
    }

    fun showBuiltInTemplateActivities(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_SHOW_BUILT_IN_TEMPLATE_ACTIVITIES] ?: false
        }
    }

    suspend fun setShowBuiltInTemplateActivities(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_SHOW_BUILT_IN_TEMPLATE_ACTIVITIES] = enabled
        }
    }

    fun enableNotifications(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_ENABLE_NOTIFICATIONS] ?: false
        }
    }

    suspend fun setEnableNotifications(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_ENABLE_NOTIFICATIONS] = enabled
        }
    }

    fun exportInternalTemplateActivities(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_EXPORT_INTERNAL_TEMPLATE_ACTIVITIES] ?: false
        }
    }

    suspend fun setExportInternalTemplateActivities(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_EXPORT_INTERNAL_TEMPLATE_ACTIVITIES] = enabled
        }
    }

    fun exportCompletedThings(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_EXPORT_COMPLETED_THINGS] ?: true
        }
    }

    suspend fun setExportCompletedThings(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_EXPORT_COMPLETED_THINGS] = enabled
        }
    }

    fun showCompletedActivities(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_SHOW_COMPLETED_ACTIVITIES] ?: false
        }
    }

    suspend fun setShowCompletedActivities(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_SHOW_COMPLETED_ACTIVITIES] = enabled
        }
    }

    fun exportRetiredComponents(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_EXPORT_RETIRED_COMPONENTS] ?: false
        }
    }

    suspend fun setExportRetiredComponents(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_EXPORT_RETIRED_COMPONENTS] = enabled
        }
    }

    fun automaticallyUpdateComponentDistanceFromBike(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_AUTOMATICALLY_UPDATE_COMPONENT_DISTANCE_FROM_BIKE] ?: true
        }
    }

    suspend fun setAutomaticallyUpdateComponentDistanceFromBike(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_AUTOMATICALLY_UPDATE_COMPONENT_DISTANCE_FROM_BIKE] = enabled
        }
    }

    fun automaticallyExportData(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_ENABLED] ?: false
        }
    }

    suspend fun setAutomaticallyExportData(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_ENABLED] = enabled
        }
    }

    fun automaticallyPruneExportedData(): Flow<Boolean> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_PRUNING_ENABLED] ?: true
        }
    }

    suspend fun setAutomaticallyPruneExportedData(enabled: Boolean) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_PRUNING_ENABLED] = enabled
        }
    }

    fun automaticExportPruningLimit(): Flow<Int> {
        return userDataStorePreferences.data.catch {
            emit(emptyPreferences())
        }.map { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_PRUNING_LIMIT] ?: 10
        }
    }

    suspend fun setAutomaticExportPruningLimit(limit: Int) {
        userDataStorePreferences.edit { preferences ->
            preferences[KEY_AUTOMATIC_EXPORT_PRUNING_LIMIT] = limit
        }
    }

    suspend fun dumpAllPreferences() {
        Log.d("UPM", "Dumping preferences:")
        Log.d("UPM", userDataStorePreferences.data.firstOrNull().toString())
    }

    private companion object {
        // actual preferences
        val KEY_THEME = stringPreferencesKey(name = "preference_theme")
        val KEY_SI_OR_WEIRDO = stringPreferencesKey(name = "preference_distances_use_km")
        val KEY_ADVANCED_FUNCTIONALITY =
            booleanPreferencesKey(name = "preferences_advanced_functionality")
        val KEY_STRAVA_ENABLED = booleanPreferencesKey(name = "preference_strava_enabled")
        val KEY_INTERVALS_ENABLED = booleanPreferencesKey(name = "preference_intervals_enabled")
        val KEY_TEMPLATE_ACTIVITY_LANGUAGE =
            stringPreferencesKey(name = "preference_template_activity_language")
        val KEY_AUTOMATICALLY_UPDATE_TEMPLATES_ON_APP_START =
            booleanPreferencesKey(name = "preference_automatically_update_template_activities_on_app_start")
        val KEY_UPDATE_BIKE_DISTANCES_PERIODICALLY =
            booleanPreferencesKey(name = "preference_update_bike_distances_periodically")
        val KEY_SHOW_BUILT_IN_TEMPLATE_ACTIVITIES =
            booleanPreferencesKey(name = "preference_show_built_in_template_activities")
        val KEY_ENABLE_NOTIFICATIONS =
            booleanPreferencesKey(name = "preference_enable_notifications")
        val KEY_EXPORT_INTERNAL_TEMPLATE_ACTIVITIES =
            booleanPreferencesKey(name = "preference_export_internal_template_activities")
        val KEY_EXPORT_COMPLETED_THINGS =
            booleanPreferencesKey(name = "preference_export_completed_things")
        val KEY_AUTOMATIC_EXPORT_ENABLED =
            booleanPreferencesKey(name = "preference_automatic_export_enabled")
        val KEY_AUTOMATIC_EXPORT_PRUNING_ENABLED =
            booleanPreferencesKey(name = "preferences_automatic_export_pruning_enabled")
        val KEY_AUTOMATIC_EXPORT_PRUNING_LIMIT =
            intPreferencesKey(name = "preferences_automatic_export_pruning_limit")

        // persistence for things that happen or from other systems
        val KEY_STRAVA_AUTH_STATE = stringPreferencesKey(name = "persisted_auth_state")
        val KEY_STRAVA_ALLOWED_ALL_PROFILE_DATA =
            booleanPreferencesKey(name = "persisted_strava_allowed_all_profile_data")
        val KEY_INTERVALS_AUTH_STATE = stringPreferencesKey(name = "persisted_intervals_auth_state")
        val KEY_INTERVALS_ALLOWED_READ_ATHLETE_SETTINGS =
            booleanPreferencesKey(name = "persisted_intervals_allowed_read_athlete_settings")
        val KEY_INTERVALS_SI_OR_WEIRDO =
            stringPreferencesKey(name = "persisted_interval_distances_use_km")
        val KEY_SHOW_COMPLETED_ACTIVITIES =
            booleanPreferencesKey(name = "persisted_show_completed_activities")
        val KEY_EXPORT_RETIRED_COMPONENTS = booleanPreferencesKey(name = "persisted_export_retired_components")
        val KEY_AUTOMATICALLY_UPDATE_COMPONENT_DISTANCE_FROM_BIKE = booleanPreferencesKey(name = "persisted_automatically_update_component_distance_from_bike")
    }
}