package com.exner.tools.jkbikemechanicaldisasterprevention.network.strava

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.core.net.toUri
import com.exner.tools.jkbikemechanicaldisasterprevention.MainActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.network.strava.models.StravaDetailedAthlete
import com.exner.tools.jkbikemechanicaldisasterprevention.network.strava.models.StravaDetailedGear
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import net.openid.appauth.AuthState
import net.openid.appauth.AuthorizationException
import net.openid.appauth.AuthorizationRequest
import net.openid.appauth.AuthorizationResponse
import net.openid.appauth.AuthorizationService
import net.openid.appauth.AuthorizationServiceConfiguration
import net.openid.appauth.GrantTypeValues
import net.openid.appauth.ResponseTypeValues
import net.openid.appauth.TokenRequest
import okhttp3.Call
import okhttp3.Callback
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import java.io.IOException
import javax.inject.Inject
import javax.inject.Singleton

private const val TAG = "StravaApiManager"

@Singleton
class StravaApiManager @Inject constructor(
    val userPreferencesManager: UserPreferencesManager
) {
    private var serviceConfig: AuthorizationServiceConfiguration? = null
    private var authState: AuthState? = null

    private var allowedToReadAllProfileData = false

    init {
        Log.d(TAG, "init $this")
        // all we need for the OAuth2 flow for Strava
        serviceConfig = AuthorizationServiceConfiguration(
            "https://www.strava.com/oauth/mobile/authorize".toUri(),
            "https://jkbike.net/proxy/strava_token.php".toUri()
        )
        CoroutineScope(Dispatchers.Default).launch {
            // try to retrieve authState
            val tempAuthStateString = userPreferencesManager.stravaAuthStateAsString().firstOrNull()
            if (!tempAuthStateString.isNullOrEmpty()) {
                authState = AuthState.jsonDeserialize(tempAuthStateString)
                Log.d(TAG, "Got AuthState from preferences: $authState")
            } else {
                // didn't find any, so make a new one
                authState = AuthState(serviceConfig!!)
            }
            userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
            // try to retrieve permissions
            allowedToReadAllProfileData =
                userPreferencesManager.stravaAllowedAllProfileData().firstOrNull() == true
            Log.d(
                TAG,
                "Allowed to read all profile data $allowedToReadAllProfileData"
            )
        }
    }

    fun isConnected(): Boolean {
//        Log.d(TAG, "isConnected()? ${authState?.isAuthorized}")
        return authState?.isAuthorized == true
    }

    fun isAllowedToReadAllProfileData(): Boolean {
        return allowedToReadAllProfileData
    }

    fun createJKAuthRequest(
        clientId: String,
        callBack: (AuthorizationRequest) -> Unit
    ) {
        if (clientId.isNotEmpty()) {
            val authRequestBuilder = AuthorizationRequest.Builder(
                serviceConfig!!,
                clientId,
                ResponseTypeValues.CODE,
                "https://jkbike.net/authredirect/".toUri()
            )
            val additionalParameters: MutableMap<String, String> = HashMap()
            additionalParameters.put(
                "approval_prompt",
                "force"
            )
            val authRequest =
                authRequestBuilder
                    .setScope("read,profile:read_all")
                    .setAdditionalParameters(
                        additionalParameters
                    )
                    .build()
            callBack(authRequest)
        }
    }

    fun useCode(
        context: Context,
        code: String,
        clientId: String,
        successCallback: (Boolean) -> Unit
    ) {
        if (code.isNotEmpty() && serviceConfig != null && clientId.isNotEmpty()) {
            Log.d(TAG, "We have a code! $code")
            val authService = AuthorizationService(context)
            val tokenRequest = TokenRequest.Builder(
                serviceConfig!!,
                clientId,
            )
                .setRedirectUri("https://jkbike.net/authredirect/".toUri())
                .setGrantType(GrantTypeValues.AUTHORIZATION_CODE)
                .setAuthorizationCode(code)
                .build()
            authService.performTokenRequest(
                tokenRequest,
            ) { response, ex ->
                Log.d(TAG, "authState $authState, resp $response, ex $ex")
                authState?.update(response, ex)
                CoroutineScope(Dispatchers.Default).launch {
                    Log.d(TAG, "Storing AuthState to preferences: $authState")
                    userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
                }
                if (response != null) { // yay!
                    Log.d(TAG, "We have a token response $response")
                    Log.d(TAG, "authState now connected ${authState?.isAuthorized}")
                    successCallback(authState?.isAuthorized == true)
                } else { // nay
                    Log.d(TAG, "Oops $ex")
                }
            }
        }
    }

    fun applyAllowedScopes(scopeString: String) {
        val scopeList = scopeString.split(",")
        // looking for "read,profile:read_all,activity:read_all"
        allowedToReadAllProfileData = scopeList.contains("profile:read_all")
        Log.d(TAG, "Allowed to read all profile data $allowedToReadAllProfileData")
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(
                TAG,
                "Storing permissions: $allowedToReadAllProfileData"
            )
            userPreferencesManager.setStravaAllowedAllProfileData(allowedToReadAllProfileData)
        }
        if (!scopeList.contains("read")) {
            Log.w(TAG, "Strava scope is missing 'read'. That is an error!")
            disconnect()
        }
    }

    fun connect(
        context: Context,
        clientId: String
    ) {
        if (authState?.isAuthorized == true) {
            Log.d(TAG, "Connection to Strava is authorized.")
        } else {
            createJKAuthRequest(
                clientId
            ) { authRequest ->
                Log.d(TAG, "Preparing auth request for Strava....")
                Log.d(
                    TAG,
                    "authRequest ${authRequest.configuration.authorizationEndpoint} / ${authRequest.configuration.tokenEndpoint} / ${authRequest.clientId} / ${authRequest.responseType} / ${authRequest.redirectUri}"
                )
                Log.d(
                    TAG,
                    "authRequest ${authRequest.additionalParameters} / ${authRequest.toUri()}, now making authService..."
                )
                val authService = AuthorizationService(context)

                Log.d(TAG, "Now performing authorization...")
                authService.performAuthorizationRequest(
                    authRequest,
                    PendingIntent.getActivity(
                        context, 0, Intent(context, MainActivity::class.java),
                        PendingIntent.FLAG_IMMUTABLE
                    ),
                    PendingIntent.getActivity(
                        context, 0, Intent(context, MainActivity::class.java),
                        PendingIntent.FLAG_IMMUTABLE
                    )
                )
            }
        }
    }

    fun disconnect() {
        // try recreating a fresh authState
        authState = AuthState(serviceConfig!!)
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Storing AuthState to preferences: $authState")
            userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
            userPreferencesManager.setStravaAllowedAllProfileData(false)
        }
    }

    fun updateAuthState(
        authResponse: AuthorizationResponse?,
        authException: AuthorizationException?
    ) {
        Log.d(TAG, "Updating authState $authResponse / $authException")
        authState?.update(authResponse, authException)
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Storing AuthState to preferences: $authState")
            userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
        }
    }

    fun retrieveLoggedInAthlete(
        context: Context,
        onResult: (StravaDetailedAthlete?) -> Unit,
        onError: (Int) -> Unit
    ) {
        val authService = AuthorizationService(context)

        Log.d(TAG, "Want to retrieve logged-in athletes, authState ${authState?.isAuthorized}")
        Log.d(TAG, "  Quick check, are we allowed? $allowedToReadAllProfileData")
        if (allowedToReadAllProfileData) {
            authState?.performActionWithFreshTokens(
                authService
            ) { accessToken, idToken, ex ->
                Log.d(TAG, "aSpAWFT results: $accessToken, $idToken, $ex")
                if (ex != null) {
                    Log.e(TAG, "Unable to acquire refresh_token $ex")
                    return@performActionWithFreshTokens
                }
                CoroutineScope(Dispatchers.Default).launch {
                    Log.d(TAG, "Storing AuthState to preferences: $authState")
                    userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
                }
                val client = OkHttpClient()
                val request = Request.Builder()
                    .url("https://www.strava.com/api/v3/athlete")
                    .addHeader("Authorization", "Bearer $accessToken")
                    .build()
                Log.d(TAG, "Requesting logged-in athlete... $request")
                client.newCall(request).enqueue(object : Callback {
                    override fun onFailure(call: Call, e: IOException) {
                        Log.d(TAG, "Failed getting logged-in athlete: ${e.message}")
                        onError(500) // generic because we do not know
                    }

                    override fun onResponse(call: Call, response: Response) {
                        Log.d(TAG, "Received result: $response")
                        val jsonContent = response.body?.string()
                        Log.d(TAG, "  json: $jsonContent")
                        if (response.code == 200) {
                            if (!jsonContent.isNullOrEmpty()) {
                                val decoder = Json { ignoreUnknownKeys = true }
                                val tempDecoded =
                                    decoder.decodeFromString<StravaDetailedAthlete>(jsonContent)
                                Log.d(TAG, "  decoded: $tempDecoded")
                                onResult(tempDecoded)
                            }
                        } else {
                            Log.w(TAG, "${response.code} error: $jsonContent")
                            onError(response.code)
                        }
                        response.close()
                    }
                })
            }
        } else {
            Log.w(TAG, "We're not allowed to read the data we need!")
            onError(403) // unauthorized
        }
    }

    fun retrieveGear(
        context: Context,
        gearId: String,
        onResult: (StravaDetailedGear) -> Unit,
        onError: (Int) -> Unit
    ) {
        val authService = AuthorizationService(context)

        Log.d(TAG, "Want to retrieve gear $gearId, authState ${authState?.isAuthorized}")
        Log.d(TAG, "  Quick check, are we allowed? $allowedToReadAllProfileData")
        if (allowedToReadAllProfileData) {
            authState?.performActionWithFreshTokens(
                authService
            ) { accessToken, idToken, ex ->
                Log.d(TAG, "aSpAWFT results: $accessToken, $idToken, $ex")
                if (ex != null) {
                    Log.e(TAG, "Unable to acquire refresh_token $ex")
                    return@performActionWithFreshTokens
                }
                CoroutineScope(Dispatchers.Default).launch {
                    Log.d(TAG, "Storing AuthState to preferences: $authState")
                    userPreferencesManager.setStravaAuthStateAsString(authState?.jsonSerializeString())
                }
                val client = OkHttpClient()
                val request = Request.Builder()
                    .url("https://www.strava.com/api/v3/gear/$gearId")
                    .addHeader("Authorization", "Bearer $accessToken")
                    .build()
                Log.d(TAG, "Requesting gear $gearId... $request")
                client.newCall(request).enqueue(object : Callback {
                    override fun onFailure(call: Call, e: IOException) {
                        Log.d(TAG, "Failed getting gear $gearId: ${e.message}")
                        onError(500) // generic because we do not know
                    }

                    override fun onResponse(call: Call, response: Response) {
                        Log.d(TAG, "Received result: $response")
                        val jsonContent = response.body?.string()
                        Log.d(TAG, "  json: $jsonContent")
                        if (response.code == 200) {
                            if (!jsonContent.isNullOrEmpty()) {
                                val decoder = Json { ignoreUnknownKeys = true }
                                val tempDecoded =
                                    decoder.decodeFromString<StravaDetailedGear>(jsonContent)
                                Log.d(TAG, "  decoded: $tempDecoded")
                                onResult(tempDecoded)
                            }
                        } else {
                            Log.w(TAG, "${response.code} error: $jsonContent")
                            onError(response.code)
                        }
                        response.close()
                    }
                })
            }
        } else {
            Log.w(TAG, "We're not allowed to read the data we need!")
            onError(403) // unauthorized
        }
    }

}
