package com.exner.tools.jkbikemechanicaldisasterprevention.network.intervals

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.util.Log
import androidx.core.net.toUri
import com.exner.tools.jkbikemechanicaldisasterprevention.MainActivity
import com.exner.tools.jkbikemechanicaldisasterprevention.network.intervals.models.IntervalsAthlete
import com.exner.tools.jkbikemechanicaldisasterprevention.network.intervals.models.IntervalsGear
import com.exner.tools.jkbikemechanicaldisasterprevention.preferences.UserPreferencesManager
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import net.openid.appauth.AuthState
import net.openid.appauth.AuthorizationException
import net.openid.appauth.AuthorizationRequest
import net.openid.appauth.AuthorizationResponse
import net.openid.appauth.AuthorizationService
import net.openid.appauth.AuthorizationServiceConfiguration
import net.openid.appauth.GrantTypeValues
import net.openid.appauth.ResponseTypeValues
import net.openid.appauth.TokenRequest
import okhttp3.OkHttpClient
import okhttp3.Request
import javax.inject.Inject
import javax.inject.Singleton

private const val TAG = "IntervalsApiManager"

@Singleton
class IntervalsApiManager @Inject constructor(
    val userPreferencesManager: UserPreferencesManager
) {
    private var serviceConfig: AuthorizationServiceConfiguration? = null
    private var authState: AuthState? = null

    private var allowedToReadAthleteSettings = false

    init {
        Log.d(TAG, "init $this")
        // all we need for the OAuth2 flow for Intervals.icu
        serviceConfig = AuthorizationServiceConfiguration(
            "https://intervals.icu/oauth/authorize".toUri(),
            "https://jkbike.net/proxy/intervals_token.php".toUri()
        )
        CoroutineScope(Dispatchers.Default).launch {
            // try to retrieve authState
            val tempAuthStateString =
                userPreferencesManager.intervalsAuthStateAsString().firstOrNull()
            if (!tempAuthStateString.isNullOrEmpty()) {
                authState = AuthState.jsonDeserialize(tempAuthStateString)
                Log.d(TAG, "Got Intervals AuthState from preferences: $authState")
            } else {
                // didn't find any, so make a new one
                authState = AuthState(serviceConfig!!)
            }
            userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
            // try to retrieve permissions
            allowedToReadAthleteSettings =
                userPreferencesManager.intervalsAllowedAthleteSettings().firstOrNull() == true
            Log.d(TAG, "Allowed to read athlete data $allowedToReadAthleteSettings")
        }
    }

    fun isConnected(): Boolean {
//        Log.d(TAG, "isConnected()? ${authState?.isAuthorized}")
        return authState?.isAuthorized == true
    }

    fun isAllowedToReadAthleteSettings(): Boolean {
        return allowedToReadAthleteSettings
    }

    // intervals auth URL:
    // https://intervals.icu/oauth/authorize?client_id=<your client id>
    //    &redirect_uri=<your redirect uri>
    //    &scope=<required scopes>
    //    &state=<optional data>

    fun createJKAuthRequest(
        clientId: String,
        callBack: (AuthorizationRequest) -> Unit
    ) {
        if (clientId.isNotEmpty()) {
            val authRequestBuilder = AuthorizationRequest.Builder(
                serviceConfig!!,
                clientId,
                ResponseTypeValues.CODE,
                "https://jkbike.net/authredirect/".toUri()
            )
            val authRequest =
                authRequestBuilder
                    .setScope("SETTINGS:READ")
                    .build()
            callBack(authRequest)
        }
    }

    fun useCode(
        context: Context,
        code: String,
        clientId: String,
        successCallback: (Boolean) -> Unit
    ) {
        if (code.isNotEmpty() && serviceConfig != null && clientId.isNotEmpty()) {
            Log.d(TAG, "We have a code! $code")
            val authService = AuthorizationService(context)
            val tokenRequest = TokenRequest.Builder(
                serviceConfig!!,
                clientId,
            )
                .setRedirectUri("https://jkbike.net/authredirect/".toUri())
                .setGrantType(GrantTypeValues.AUTHORIZATION_CODE)
                .setAuthorizationCode(code)
                .build()
            authService.performTokenRequest(
                tokenRequest,
            ) { response, ex ->
                Log.d(TAG, "authState $authState, resp $response, ex $ex")
                authState?.update(response, ex)
                CoroutineScope(Dispatchers.Default).launch {
                    Log.d(TAG, "Storing AuthState to preferences: $authState")
                    userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
                }
                if (response != null) { // yay!
                    Log.d(TAG, "We have a token response $response")
                    Log.d(TAG, "authState now connected ${authState?.isAuthorized}")
                    successCallback(authState?.isAuthorized == true)
                } else { // nay
                    Log.d(TAG, "Oops $ex")
                }
            }
        }
    }

    fun applyAllowedScopes(scopeString: String) {
        val scopeList = scopeString.split(",")
        // there will never be a scope param, so:
        allowedToReadAthleteSettings = true
        Log.d(TAG, "Allowed to read athlete data $allowedToReadAthleteSettings")
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(
                TAG,
                "Storing permissions: $allowedToReadAthleteSettings"
            )
            userPreferencesManager.setIntervalsAllowedAthleteSettings(allowedToReadAthleteSettings)
        }
        if (!allowedToReadAthleteSettings) {
            Log.w(TAG, "Intervals.icu scope is missing 'SETTINGS:READ'. That is an error!")
            disconnect()
        }
    }

    fun connect(
        context: Context,
        clientId: String
    ) {
        if (authState?.isAuthorized == true) {
            Log.d(TAG, "Connection to Intervals is authorized.")
        } else {
            createJKAuthRequest(
                clientId
            ) { authRequest ->
                Log.d(TAG, "Preparing auth request for Intervals....")
                Log.d(
                    TAG,
                    "authRequest ${authRequest.configuration.authorizationEndpoint} / ${authRequest.configuration.tokenEndpoint} / ${authRequest.clientId} / ${authRequest.responseType} / ${authRequest.redirectUri}"
                )
                Log.d(
                    TAG,
                    "authRequest ${authRequest.additionalParameters} / ${authRequest.toUri()}, now making authService..."
                )
                val authService = AuthorizationService(context)

                Log.d(TAG, "Now performing authorization...")
                authService.performAuthorizationRequest(
                    authRequest,
                    PendingIntent.getActivity(
                        context, 0, Intent(context, MainActivity::class.java),
                        PendingIntent.FLAG_IMMUTABLE
                    ),
                    PendingIntent.getActivity(
                        context, 0, Intent(context, MainActivity::class.java),
                        PendingIntent.FLAG_IMMUTABLE
                    )
                )
            }
        }
    }

    fun disconnect() {
        // try recreating a fresh authState
        authState = AuthState(serviceConfig!!)
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Storing AuthState to preferences: $authState")
            userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
            userPreferencesManager.setIntervalsAllowedAthleteSettings(false)
        }
    }

    fun updateAuthState(
        authResponse: AuthorizationResponse?,
        authException: AuthorizationException?
    ) {
        Log.d(TAG, "Updating authState $authResponse / $authException")
        authState?.update(authResponse, authException)
        CoroutineScope(Dispatchers.Default).launch {
            Log.d(TAG, "Storing AuthState to preferences: $authState")
            userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
        }
    }

    fun retrieveLoggedInAthlete(
        context: Context,
        onResult: (IntervalsAthlete?) -> Unit,
        onError: (Int) -> Unit
    ) {
        val authService = AuthorizationService(context)

        Log.d(TAG, "Want to retrieve logged-in athletes, authState ${authState?.isAuthorized}")
        Log.d(TAG, "  Quick check, are we allowed? $allowedToReadAthleteSettings")
        if (allowedToReadAthleteSettings) {
            authState?.performActionWithFreshTokens(
                authService
            ) { accessToken, idToken, ex ->
                Log.d(TAG, "aSpAWFT results: $accessToken, $idToken, $ex")
                if (ex != null) {
                    Log.e(TAG, "Unable to acquire refresh_token $ex")
                    return@performActionWithFreshTokens
                }
                CoroutineScope(Dispatchers.Default).launch {
                    Log.d(TAG, "Storing AuthState to preferences: $authState")
                    userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
                }
                CoroutineScope(Dispatchers.IO).launch {
                    val client = OkHttpClient()
                    val request = Request.Builder()
                        .url("https://intervals.icu/api/v1/athlete/0")
                        .addHeader("Authorization", "Bearer $accessToken")
                        .build()
                    Log.d(TAG, "Requesting logged-in athlete... $request")
                    val response = client.newCall(request).execute()
                    Log.d(TAG, "Received result: $response")
                    val responseBodyCopy = response.peekBody(Long.MAX_VALUE)
                    val jsonContent = responseBodyCopy.string()
                    Log.d(TAG, "  json: $jsonContent")
                    if (response.code == 200) {
                        if (!jsonContent.isNullOrEmpty()) {
                            val decoder = Json { ignoreUnknownKeys = true }
                            val tempDecoded =
                                decoder.decodeFromString<IntervalsAthlete>(jsonContent)
                            Log.d(TAG, "  decoded: $tempDecoded")
                            onResult(tempDecoded)
                        }
                    } else {
                        Log.w(TAG, "${response.code} error: $jsonContent")
                        onError(response.code)
                    }
                    response.close()
                }
            }
        } else {
            Log.w(TAG, "We're not allowed to read the data we need!")
            onError(403) // unauthorized
        }
    }

    fun retrieveGear(
        context: Context,
        onResult: (List<IntervalsGear>) -> Unit,
        onError: (Int) -> Unit
    ) {
        val authService = AuthorizationService(context)

        Log.d(
            TAG,
            "Want to retrieve gear for logged-in athlete, authState ${authState?.isAuthorized}"
        )
        try {
            Log.d(TAG, "  Quick check, are we allowed? $allowedToReadAthleteSettings")
            if (allowedToReadAthleteSettings) {
                authState?.performActionWithFreshTokens(
                    authService
                ) { accessToken, idToken, ex ->
                    Log.d(TAG, "aSpAWFT results: $accessToken, $idToken, $ex")
                    if (ex != null) {
                        Log.e(TAG, "Unable to acquire refresh_token $ex")
                        return@performActionWithFreshTokens
                    }
                    CoroutineScope(Dispatchers.Default).launch {
                        Log.d(TAG, "Storing AuthState to preferences: $authState")
                        userPreferencesManager.setIntervalsAuthStateAsString(authState?.jsonSerializeString())
                    }
                    CoroutineScope(Dispatchers.IO).launch {
                        val client = OkHttpClient()
                        val request = Request.Builder()
                            .url("https://intervals.icu/api/v1/athlete/0/gear")
                            .addHeader("Authorization", "Bearer $accessToken")
                            .build()
                        Log.d(TAG, "Requesting gear... $request")
                        val response = client.newCall(request).execute()
                        Log.d(TAG, "Received result: $response")
                        if (response.code == 200) {
                            val responseBodyCopy = response.peekBody(Long.MAX_VALUE)
                            val jsonContent = responseBodyCopy.string()
                            Log.d(TAG, "  json: $jsonContent")
                            if (!jsonContent.isNullOrEmpty()) {
                                val decoder = Json { ignoreUnknownKeys = true }
                                val tempDecoded =
                                    decoder.decodeFromString<List<IntervalsGear>>(jsonContent)
                                Log.d(TAG, "  decoded: $tempDecoded")
                                onResult(tempDecoded)
                            }
                        } else {
                            Log.w(TAG, "${response.code} error")
                            onError(response.code)
                        }
                        response.close()
                    }
                }
            } else {
                Log.w(TAG, "We're not allowed to read the data we need!")
                onError(403) // unauthorized
            }
        } catch (exception: Exception) {
            Log.w(TAG, "Exception: ${exception.message}")
        }
    }

}