package com.exner.tools.jkbikemechanicaldisasterprevention.di

import android.content.Context
import android.util.Log
import androidx.room.Room
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsDAO
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsDatabase
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsDatabaseCallback
import dagger.Module
import dagger.Provides
import dagger.hilt.InstallIn
import dagger.hilt.android.qualifiers.ApplicationContext
import dagger.hilt.components.SingletonComponent
import javax.inject.Provider
import javax.inject.Singleton

private const val TAG = "AppComponent"

@Module
@InstallIn(SingletonComponent::class)
object AppComponent {

    @Singleton
    @Provides
    fun provideDAO(kjsDatabase: KJsDatabase): KJsDAO = kjsDatabase.dataDAO()

    @Singleton
    @Provides
    fun provideDatabase(
        @ApplicationContext context: Context,
        provider: Provider<KJsDAO>
    ): KJsDatabase = Room.databaseBuilder(
        context = context.applicationContext,
        KJsDatabase::class.java,
        name = "kjs_database"
    )
        .addMigrations(MIGRATION_1_2)
        .addMigrations(MIGRATION_2_3)
        .addMigrations(MIGRATION_3_4)
        .addMigrations(MIGRATION_4_5)
        .addMigrations(MIGRATION_5_6)
        .addMigrations(MIGRATION_6_7)
        .addMigrations(MIGRATION_7_8)
        .addMigrations(MIGRATION_8_9)
        .addMigrations(MIGRATION_9_10)
        .addMigrations(MIGRATION_10_11)
        .addMigrations(MIGRATION_11_12)
        .addMigrations(MIGRATION_12_13)
        .addMigrations(MIGRATION_13_14)
        .fallbackToDestructiveMigration(dropAllTables = true)
        .addCallback(KJsDatabaseCallback(provider))
        .build()

}

val MIGRATION_1_2 = object : Migration(1, 2) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // rides first
        db.execSQL("CREATE TABLE `TempRide` (`name` TEXT NOT NULL, `level` INTEGER NOT NULL, `created_instant` INTEGER NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO `TempRide` (name, level, created_instant, uid) " +
                    "SELECT name, SUBSTR(level, 1, 1), created_instant, uid FROM `Ride`"
        )
        db.execSQL("DROP TABLE `Ride`")
        db.execSQL("ALTER TABLE `TempRide` RENAME TO `Ride`")

        // template activities next
        db.execSQL("CREATE TABLE `TempTemplateActivity` (`ride_level` INTEGER, `title` TEXT NOT NULL, `description` TEXT NOT NULL, `is_ebike_specific` INTEGER NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO `TempTemplateActivity` (ride_level, title, description, is_ebike_specific, uid) " +
                    "SELECT SUBSTR(ride_level, 1, 1), title, description, is_ebike_specific, uid FROM TemplateActivity"
        )
        db.execSQL("DROP TABLE `TemplateActivity`")
        db.execSQL("ALTER TABLE `TempTemplateActivity` RENAME TO `TemplateActivity`")

        // activity last but not least
        db.execSQL("CREATE TABLE `Temporary` (`title` TEXT NOT NULL, `description` TEXT NOT NULL, `is_completed` INTEGER NOT NULL, `bike_uid` INTEGER, `is_ebike_specific` INTEGER NOT NULL, `ride_level` INTEGER, `ride_uid` INTEGER, `created_instant` INTEGER NOT NULL, `due_date` TEXT, `done_instant` INTEGER, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO Temporary (title, description, is_completed, bike_uid, is_ebike_specific, ride_level, ride_uid, created_instant, due_date, done_instant, uid) " +
                    "SELECT title, description, is_completed, bike_uid, is_ebike_specific, SUBSTR(ride_level, 1, 1), ride_uid, created_instant, due_date, done_instant, uid FROM Activity"
        )
        db.execSQL("DROP TABLE `Activity`")
        db.execSQL("ALTER TABLE `Temporary` RENAME TO `Activity`")

        // and the view!
        db.execSQL("DROP VIEW IF EXISTS `ActivityWithBikeData`")
        db.execSQL("CREATE VIEW `ActivityWithBikeData` AS SELECT b.name as bike_name, b.uid as bike_uid, a.title as activity_title, a.description as activity_description, a.is_completed as activity_is_completed, a.ride_uid as activity_ride_uid, a.created_instant as activity_created_instant, a.due_date as activity_due_date, a.done_instant as activity_done_instant, a.is_ebike_specific as activity_is_ebike_specific, a.ride_level as activity_ride_level, a.uid as activity_uid FROM Activity a LEFT JOIN Bike b ON b.uid = a.bike_uid ORDER BY a.due_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_2_3 = object : Migration(2, 3) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Bike` ADD COLUMN `strava_gear_id` TEXT")

        db.execSQL("ALTER TABLE `Activity` ADD COLUMN `component_uid` INTEGER")

        db.execSQL("CREATE TABLE IF NOT EXISTS `Component` (`name` TEXT NOT NULL, `description` TEXT NOT NULL, `acquisition_date` TEXT, `first_use_date` TEXT, `last_check_date` TEXT, `last_check_mileage` INTEGER, `current_mileage` INTEGER, `bike_uid` INTEGER, `title_for_automatic_activities` TEXT, `wear_level` TEXT, `retirement_date` TEXT, `retirement_reason` TEXT, `check_interval_miles` INTEGER, `check_interval_days` INTEGER, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL("CREATE TABLE IF NOT EXISTS `AutomaticActivitiesGenerationLog` (`created_instant` INTEGER NOT NULL, `created_component_uid` INTEGER NOT NULL, `activity_uid` INTEGER NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        // views
        db.execSQL("DROP VIEW IF EXISTS `ActivityWithBikeData`")
        db.execSQL("CREATE VIEW `ActivityWithBikeData` AS SELECT b.name as bike_name, b.uid as bike_uid, a.title as activity_title, a.description as activity_description, a.is_completed as activity_is_completed, a.ride_uid as activity_ride_uid, a.created_instant as activity_created_instant, a.due_date as activity_due_date, a.done_instant as activity_done_instant, a.is_ebike_specific as activity_is_ebike_specific, a.ride_level as activity_ride_level, a.component_uid as activity_component_uid, a.uid as activity_uid FROM Activity a LEFT JOIN Bike b ON b.uid = a.bike_uid ORDER BY a.due_date DESC")
        db.execSQL("CREATE VIEW `ShelvedComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days,  uid FROM Component WHERE bike_uid IS NULL AND retirement_date IS NULL ORDER BY acquisition_date DESC")
        db.execSQL("CREATE VIEW `RetiredComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days,  uid FROM Component WHERE retirement_date IS NOT NULL ORDER BY retirement_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_3_4 = object : Migration(3, 4) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Bike` ADD COLUMN `is_active` INTEGER NOT NULL DEFAULT 1")
        db.execSQL("ALTER TABLE `Bike` ADD COLUMN `retirement_date` TEXT")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_4_5 = object : Migration(4, 5) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        //tables
        db.execSQL("CREATE TABLE IF NOT EXISTS `StravaSummaryGear` (`id` TEXT, `primary` INTEGER, `name` TEXT, `resource_state` INTEGER, `distance` REAL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_5_6 = object : Migration(5, 6) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables (destructive!)
        db.execSQL("DROP TABLE IF EXISTS `TemplateActivity`")
        db.execSQL("CREATE TABLE `TemplateActivity` (`ride_level` INTEGER, `title` TEXT NOT NULL, `description` TEXT NOT NULL, `is_ebike_specific` INTEGER NOT NULL, `type_of_template` TEXT NOT NULL DEFAULT 'built-in', `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_6_7 = object : Migration(6, 7) {
    // this is a bigger one bcs we went from storing Instant elements as Int to String
    // so need to read seconds since 1970 and store as ISO 8601 string
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("UPDATE `Activity` SET `created_instant` = datetime('now') WHERE `created_instant` IS NULL")
        db.execSQL("CREATE TABLE IF NOT EXISTS `TempActivity` (`title` TEXT NOT NULL, `description` TEXT NOT NULL, `is_completed` INTEGER NOT NULL, `bike_uid` INTEGER, `is_ebike_specific` INTEGER NOT NULL, `ride_level` INTEGER, `component_uid` INTEGER, `ride_uid` INTEGER, `created_instant` TEXT NOT NULL, `due_date` TEXT, `done_instant` TEXT, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO `TempActivity` (`title`, `description`, `is_completed`, `bike_uid`, `is_ebike_specific`, `ride_level`, `component_uid`, `ride_uid`, `created_instant`, `due_date`, `done_instant`, `uid`) " +
                    "SELECT `title`, `description`, `is_completed`, `bike_uid`, `is_ebike_specific`, `ride_level`, `component_uid`, `ride_uid`, datetime(`created_instant` / 1000, 'unixepoch', 'localtime'), `due_date`, datetime(`done_instant` / 1000, 'unixepoch', 'localtime'), `uid` FROM `Activity`"
        )
        db.execSQL("DROP TABLE `Activity`")
        db.execSQL("ALTER TABLE `TempActivity` RENAME TO `Activity`")
        db.execSQL("UPDATE `Ride` SET `created_instant` = datetime('now') WHERE `created_instant` IS NULL")
        db.execSQL("CREATE TABLE IF NOT EXISTS `TempRide` (`name` TEXT NOT NULL, `level` INTEGER NOT NULL, `created_instant` TEXT NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO `TempRide` (`name`, `level`, `created_instant`, `uid`) " +
                    "SELECT `name`, `level`, datetime(`created_instant` / 1000, 'unixepoch', 'localtime'), `uid` FROM `Ride`"
        )
        db.execSQL("DROP TABLE `Ride`")
        db.execSQL("ALTER TABLE `TempRide` RENAME TO `Ride`")
        db.execSQL("UPDATE `RideUidByRideLevel` SET `created_instant` = datetime('now') WHERE `created_instant` IS NULL")
        db.execSQL("CREATE TABLE IF NOT EXISTS `TempRideUidByRideLevel` (`ride_level` INTEGER NOT NULL, `ride_uid` INTEGER NOT NULL, `created_instant` TEXT NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL(
            "INSERT INTO `TempRideUidByRideLevel` (`ride_level`, `ride_uid`, `created_instant`, `uid`) " +
                    "SELECT `ride_level`, `ride_uid`, datetime(`created_instant` / 1000, 'unixepoch', 'localtime'), `uid` FROM `RideUidByRideLevel`"
        )
        db.execSQL("DROP TABLE `RideUidByRideLevel`")
        db.execSQL("ALTER TABLE `TempRideUidByRideLevel` RENAME TO `RideUidByRideLevel`")
        db.execSQL("DROP TABLE IF EXISTS `AutomaticActivitiesGenerationLog`") // is likely empty anyway
        db.execSQL("CREATE TABLE IF NOT EXISTS `AutomaticActivitiesGenerationLog` (`created_instant` TEXT NOT NULL, `created_component_uid` INTEGER NOT NULL, `activity_uid` INTEGER NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_7_8 = object : Migration(7, 8) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Bike` ADD COLUMN `intervals_gear_id` TEXT")
        db.execSQL("CREATE TABLE IF NOT EXISTS `IntervalsSummaryGear` (`id` TEXT, `name` TEXT, `distance` REAL, `primary` INTEGER, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_8_9 = object : Migration(8, 9) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("CREATE TABLE IF NOT EXISTS `IntervalsSummaryGear` (`id` TEXT, `name` TEXT, `distance` REAL, `primary` INTEGER, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")
        db.execSQL("DROP TABLE IF EXISTS `AutomaticActivitiesGenerationLog`")
        db.execSQL("CREATE TABLE IF NOT EXISTS `BackgroundWorkLog` (`created_instant` TEXT NOT NULL, `source` TEXT NOT NULL, `created_component_uid` INTEGER, `activity_uid` INTEGER, `description` TEXT NOT NULL, `uid` INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL)")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_9_10 = object : Migration(9, 10) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Ride` ADD COLUMN `is_completed` INTEGER NOT NULL DEFAULT 0")
        db.execSQL("ALTER TABLE `Ride` ADD COLUMN `completed_instant` TEXT")
        db.execSQL("ALTER TABLE `Ride` ADD COLUMN `bike_uid` INTEGER")

        // views
        db.execSQL("DROP VIEW IF EXISTS `ActivityWithBikeData`")
        db.execSQL("CREATE VIEW `ActivityWithBikeAndRideData` AS SELECT b.name as bike_name, b.uid as bike_uid, b.is_active as bike_is_active, r.name as ride_name, r.level as ride_level, r.is_completed as ride_is_completed, r.completed_instant as ride_completed_instant, a.title as activity_title, a.description as activity_description, a.is_completed as activity_is_completed, a.ride_uid as activity_ride_uid, a.created_instant as activity_created_instant, a.due_date as activity_due_date, a.done_instant as activity_done_instant, a.is_ebike_specific as activity_is_ebike_specific, a.ride_level as activity_ride_level, a.component_uid as activity_component_uid, a.uid as activity_uid FROM Activity a LEFT JOIN Bike b ON b.uid = a.bike_uid LEFT JOIN Ride r ON r.uid = a.ride_uid ORDER BY a.due_date DESC")
        db.execSQL("CREATE VIEW `RideWithBikeAndActivityTotals` AS SELECT r.name as ride_name, r.level as ride_level, r.created_instant as ride_created_instant, r.is_completed as ride_is_completed, r.completed_instant as ride_completed_instant, r.uid as ride_uid, b.name as bike_name, b.uid as bike_uid, b.is_active as bike_is_active, COUNT(DISTINCT a.uid) as activity_number_total, COUNT(DISTINCT a.uid) FILTER(WHERE a.is_completed = 1) as activity_number_completed FROM Ride r LEFT JOIN Bike b ON b.uid = r.bike_uid LEFT JOIN Activity a ON r.uid = a.ride_uid GROUP By r.uid ORDER BY datetime(r.created_instant) DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_10_11 = object : Migration(10, 11) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Component` ADD COLUMN `notify_when_activity_created` INTEGER NOT NULL DEFAULT 0")

        // views
        db.execSQL("DROP VIEW IF EXISTS `ShelvedComponents`")
        db.execSQL("CREATE VIEW `ShelvedComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created,  uid FROM Component WHERE bike_uid IS NULL AND retirement_date IS NULL ORDER BY acquisition_date DESC")
        db.execSQL("DROP VIEW IF EXISTS `RetiredComponents`")
        db.execSQL("CREATE VIEW `RetiredComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created,  uid FROM Component WHERE retirement_date IS NOT NULL ORDER BY retirement_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_11_12 = object : Migration(11, 12) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // views
        db.execSQL("DROP VIEW IF EXISTS `ActivitiesButNotForCompletedRides`")
        db.execSQL("CREATE VIEW `ActivitiesButNotForCompletedRides` AS SELECT a.title as title, a.description as description, a.is_completed as is_completed, a.bike_uid as bike_uid, a.is_ebike_specific as is_ebike_specific, a.ride_level as ride_level, a.component_uid as component_uid, a.ride_uid as ride_uid, a.created_instant as created_instant, a.due_date as due_date, a.done_instant as done_instant, a.uid as uid FROM Activity a LEFT JOIN Ride r ON r.uid = a.ride_uid WHERE a.ride_uid IS NULL OR r.is_completed = 0 ORDER BY a.due_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_12_13 = object : Migration(12, 13) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        //tables
        db.execSQL("ALTER TABLE `Component` ADD COLUMN `last_check_notification_mileage` INTEGER")
        db.execSQL("ALTER TABLE `Component` ADD COLUMN `last_check_notification_date` TEXT")

        //views
        db.execSQL("DROP VIEW IF EXISTS `ShelvedComponents`")
        db.execSQL("CREATE VIEW `ShelvedComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created, last_check_notification_mileage, last_check_notification_date,  uid FROM Component WHERE bike_uid IS NULL AND retirement_date IS NULL ORDER BY acquisition_date DESC")
        db.execSQL("DROP VIEW IF EXISTS `RetiredComponents`")
        db.execSQL("CREATE VIEW `RetiredComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created, last_check_notification_mileage, last_check_notification_date,  uid FROM Component WHERE retirement_date IS NOT NULL ORDER BY retirement_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_13_14 = object : Migration(13, 14) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")
        // tables
        db.execSQL("ALTER TABLE `Component` ADD COLUMN `mileage_offset` INTEGER")

        //views
        db.execSQL("DROP VIEW IF EXISTS `ShelvedComponents`")
        db.execSQL("CREATE VIEW `ShelvedComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, mileage_offset, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created, last_check_notification_mileage, last_check_notification_date,  uid FROM Component WHERE bike_uid IS NULL AND retirement_date IS NULL ORDER BY acquisition_date DESC")
        db.execSQL("DROP VIEW IF EXISTS `RetiredComponents`")
        db.execSQL("CREATE VIEW `RetiredComponents` AS SELECT name, description, acquisition_date, first_use_date, last_check_date, last_check_mileage, current_mileage, mileage_offset, bike_uid, title_for_automatic_activities, wear_level, retirement_date, retirement_reason, check_interval_miles, check_interval_days, notify_when_activity_created, last_check_notification_mileage, last_check_notification_date,  uid FROM Component WHERE retirement_date IS NOT NULL ORDER BY retirement_date DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}

val MIGRATION_14_15 = object : Migration(14, 15) {
    override fun migrate(db: SupportSQLiteDatabase) {
        Log.d(TAG, "Migrating $startVersion -> $endVersion...")

        //views
        db.execSQL("DROP VIEW IF EXISTS `RideWithBikeAndActivityTotals`")
        db.execSQL("CREATE VIEW `RideWithBikeAndActivityTotals` AS SELECT r.name as ride_name, r.level as ride_level, r.created_instant as ride_created_instant, r.is_completed as ride_is_completed, r.completed_instant as ride_completed_instant, r.uid as ride_uid, b.name as bike_name, b.uid as bike_uid, b.is_active as bike_is_active, COUNT(DISTINCT a.uid) as activity_number_total, COUNT(DISTINCT (CASE WHEN a.is_completed = 1 THEN a.uid END)) as activity_number_completed FROM Ride r LEFT JOIN Bike b ON b.uid = r.bike_uid LEFT JOIN Activity a ON r.uid = a.ride_uid GROUP By r.uid ORDER BY r.created_instant DESC")

        Log.d(TAG, "Migration $startVersion -> $endVersion done.")
    }
}