package com.exner.tools.jkbikemechanicaldisasterprevention.database.tools

import android.content.ContentValues
import android.content.Context
import android.os.Environment
import android.provider.MediaStore
import android.util.Log
import com.exner.tools.jkbikemechanicaldisasterprevention.database.KJsRepository
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime
import kotlinx.serialization.json.Json
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

class Export(
    val repository: KJsRepository,
    val context: Context
) {
    @OptIn(ExperimentalStdlibApi::class, ExperimentalTime::class)
    fun commitExport(
        exportType: String = "export",
        exportBuiltInTemplates: Boolean = false,
        exportCompletedActivitiesAndRides: Boolean = true,
        successCallback: (() -> Unit)? = null,
        failureCallback: ((String) -> Unit)? = null
    ) {
        CoroutineScope(Dispatchers.Default).launch {
            try {
                // load data
                val data = bundleData(
                    exportBuiltInTemplates = exportBuiltInTemplates,
                    exportCompletedActivitiesAndRides = exportCompletedActivitiesAndRides
                ) // collect all data from the DB
                val overallJsonString = Json.encodeToString(data)
                Log.d("Export", "Collected data: $overallJsonString")
                val now =
                    Clock.System.now().toLocalDateTime(TimeZone.Companion.currentSystemDefault())
                val timeComponentForExportFileName = now.toString()
                // now write it to the Downloads folder
                val templateComponent = if (exportBuiltInTemplates) {
                    "all"
                } else {
                    "customOnly"
                }
                val rideComponent = if (exportCompletedActivitiesAndRides) {
                    "all"
                } else {
                    "noClosed"
                }
                val typeComponent = if (exportType == "export") {
                    "export"
                } else {
                    "backup"
                }
                val contentValues = ContentValues().apply {
                    put(
                        MediaStore.MediaColumns.DISPLAY_NAME,
                        "jkbike-$typeComponent-$templateComponent-$rideComponent-$timeComponentForExportFileName"
                    )
                    put(MediaStore.MediaColumns.MIME_TYPE, "application/json")
                    put(MediaStore.MediaColumns.RELATIVE_PATH, Environment.DIRECTORY_DOWNLOADS)
                }
                val resolver = context.contentResolver
                val uri = resolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)
                if (uri != null) {
                    resolver.openOutputStream(uri).use { stream ->
                        stream?.write(overallJsonString.encodeToByteArray())
                    }
                }
                Log.d("Export", "Exported successfully")
                // notify
                if (successCallback != null) {
                    successCallback()
                }
            } catch (exception: Exception) {
                Log.d("Export", "Something untoward happened: ${exception.message}")
                if (failureCallback != null) {
                    failureCallback(exception.message ?: "Failure (unknown)")
                }
            }
        }
    }

    private suspend fun bundleData(
        exportBuiltInTemplates: Boolean = false,
        exportCompletedActivitiesAndRides: Boolean = true
    ): RootData {
        val listOfAllBikes = repository.getAllBikes()
        val listOfAllActivities = if (exportCompletedActivitiesAndRides) {
            repository.getAllActivities()
        } else {
            repository.getAllActivitiesButNotForClosedRides()
        }
        val listOfTemplateActivities =
            repository.getAllTemplateActivities().filter { templateActivity ->
                exportBuiltInTemplates || templateActivity.typeOfTemplate == TemplateActivityType.CUSTOM
            }
        val listOfAllRides = repository.getAllRides().filter { ride ->
            exportCompletedActivitiesAndRides || ride.completed == false
        }
        val listOfComponents = repository.getAllComponents()
        val data = RootData(
            listOfAllBikes,
            listOfAllActivities,
            listOfTemplateActivities,
            listOfComponents,
            listOfAllRides
        )
        return data
    }
}